/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.workbench.internal;

import java.io.PrintWriter;
import java.io.StringWriter;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.TrayDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.simantics.PlatformException;
import org.simantics.SimanticsPlatform;
import org.simantics.db.Session;
import org.simantics.db.SessionManager;
import org.simantics.db.SessionReference;
import org.simantics.db.event.SessionEvent;
import org.simantics.db.event.SessionListener;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.management.ISessionContextChangedListener;
import org.simantics.db.management.ISessionContextProvider;
import org.simantics.db.management.SessionContextChangedEvent;
import org.simantics.db.service.LifecycleSupport;
import org.simantics.utils.ui.ErrorLogger;

/**
 * SessionWatchdog is essentially a session manager event listener that catches
 * exceptions happening within the session connection. Currently the only
 * procedure to take when a connection exception comes is to just kill the
 * application since there's really no client-visible way of recovering from the
 * errors.
 * 
 * <p>
 * Note that SessionWatchdog is currently for use with the eclipse Workbench UI.
 * It is not intended for headless application use.
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
public class SessionWatchdog implements SessionListener, ISessionContextChangedListener {

    SessionManager currentManager;

    public void attach(ISessionContextProvider scp) {
        scp.addContextChangedListener(this);
    }

    private void attach(Session session) {
        SessionManager sessionManager = null;
        synchronized (this) {
            sessionManager = toSessionManager(session);
            if (sessionManager == currentManager)
                return;
        }

        if (currentManager != null) {
            currentManager.removeSessionListener(this);
        }
        this.currentManager = sessionManager;
        if (sessionManager != null) {
            sessionManager.addSessionListener(this);
        }
    }

    private SessionManager toSessionManager(Session s) {
        if (s == null)
            return null;
        LifecycleSupport ls = s.peekService(LifecycleSupport.class);
        if (ls == null)
            return null;
        return ls.getSessionManager();
    }

    @Override
    public void sessionContextChanged(SessionContextChangedEvent event) {
        ISessionContext sc = event.getNewValue();
        attach((sc != null) ? sc.getSession() : (Session) null);
    }

    @Override
    public void sessionOpened(SessionEvent e) {
        //System.err.println(this + ": session opened: " + e.getSession() + "\n" + e.getCause());
    }

    @Override
    public void sessionClosed(SessionEvent e) {
        //System.err.println(this + ": session closed: " + e.getSession() + "\n" + e.getCause());
        if (e.getCause() != null) {
            ErrorLogger.defaultLogError(e.getCause());

            // TODO: attempt reconnection, this is the right place for that.
            reconnect(e.getSession());
        }
    }

    private void reconnect(Session deadSession) {
//        Session s = deadSession;
//
//        LifecycleSupport ls = s.getService(LifecycleSupport.class);
//        SessionReference sr = ls.getSessionReference();

        // TODO: make it possible to get the old user authenticator agent from the dead session!
//        SessionUserSupport sus = s.getService(SessionUserSupport.class);
//        String userName = sus.getUserName();
//        String localDigest = sus.getUserLocalDigest();
//        String remoteDigest = sus.getUserRemoteDigest();
//        currentManager.createSession(sr,
//                UserAuthenticationAgents.staticAgent(
//                        UserAuthenticators.byNameAndDigest(userName, localDigest, remoteDigest)));
    }

    @Override
    public void sessionException(SessionEvent e) {
        // First things first, log the error.
        ErrorLogger.defaultLogError(e.getCause());

        Session s = e.getSession();
        LifecycleSupport ls = s.getService(LifecycleSupport.class);
        SessionReference sr = ls.getSessionReference();

        Display display = Display.getDefault();
        display.asyncExec(dialogOpener(e.getCause(), sr));
    }

    private Runnable dialogOpener(final Throwable cause, final SessionReference sr) {
        return new Runnable() {
            @Override
            public void run() {
                IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
                if (window == null)
                    return;
                Shell shell = window.getShell();
                if (shell == null)
                    return;

                DatabaseConnectionProblemDialog d = new DatabaseConnectionProblemDialog(shell, cause, sr);
                d.setBlockOnOpen(true);
                d.open();
                // The user chose to continue, but this will with 99.9%
                // certainty just result in a program that is stuck.
                return;
            }
        };
    }

    private static void shutdown() {
        // Try to kill all active servers nicely and possibly with force
        // if nice doesn't work.
        try {
            SimanticsPlatform.INSTANCE.shutdown(null);
        } catch (PlatformException e) {
            e.printStackTrace();
        }

        // The workbench won't close if we try to close it using
        // PlatformUI.getWorkbench().close() so we'll just kill the
        // process for now.
        System.exit(-2);
    }

    class DatabaseConnectionProblemDialog extends TrayDialog {

        Throwable cause;
        SessionReference sr;

        public DatabaseConnectionProblemDialog(Shell shell, Throwable cause, SessionReference sr) {
            super(shell);
            this.cause = cause;
            this.sr = sr;

            setShellStyle(SWT.TITLE | SWT.APPLICATION_MODAL | SWT.RESIZE | getDefaultOrientation());
        }

        @Override
        protected boolean isResizable() {
            return true;
        }

        @Override
        protected void configureShell(Shell newShell) {
            super.configureShell(newShell);
            newShell.setSize(700, 600);
            newShell.setText("Database Connection Problem");
        }

        @Override
        protected void createButtonsForButtonBar(Composite parent) {
            createButton(parent, IDialogConstants.PROCEED_ID, "Continue", false);
            createButton(parent, IDialogConstants.CLOSE_ID, IDialogConstants.CLOSE_LABEL, true);
        }

        @Override
        protected void buttonPressed(int buttonId) {
            if (IDialogConstants.CLOSE_ID == buttonId) {
                shutdown();
            } else if (IDialogConstants.PROCEED_ID == buttonId) {
                // Continue == CANCEL.
                cancelPressed();
            }
        }

        @Override
        protected Control createDialogArea(Composite parent) {
            Composite c = (Composite) super.createDialogArea(parent);

            Text text = new Text(c, SWT.MULTI | SWT.READ_ONLY | SWT.WRAP);
            StringBuilder sb = new StringBuilder();
            sb.append("Our apologies, it seems that the database connection to used by the workbench has been unexpectedly disconnected. We are working on these stability problems.");
            sb.append("\n\n");
            sb.append("Session Reference was: " + sr);
            sb.append("\n\n");
            sb.append("Unfortunately nothing can be done at this point to recover your unsaved work. The program will be forcibly closed if you select close. You may select continue to attempt recovery but be warned that this will most likely not achieve anything useful.");
            sb.append("\n\n");
            sb.append("TODO: implement a possibility to attempt reconnection here if the database is still up and running!");
            text.setText(sb.toString());
            GridDataFactory.fillDefaults().grab(true, true).applyTo(text);

            Label causeLabel = new Label(c, SWT.NONE);
            causeLabel.setText("Exception:");
            GridDataFactory.fillDefaults().grab(true, false).applyTo(causeLabel);

            Text causeText = new Text(c, SWT.MULTI | SWT.READ_ONLY | SWT.WRAP | SWT.BORDER);
            StringWriter sw = new StringWriter();
            cause.printStackTrace(new PrintWriter(sw));
            causeText.setText(sw.toString());
            GridDataFactory.fillDefaults().grab(true, true).applyTo(causeText);

            return c;
        }

    }

}
