package org.simantics.ui.workbench.handler;

import java.lang.reflect.InvocationTargetException;
import java.util.Collection;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.handlers.HandlerUtil;
import org.eclipse.ui.progress.IProgressService;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;
import org.simantics.simulation.ontology.SimulationResource;
import org.simantics.utils.ui.ExceptionUtils;
import org.simantics.utils.ui.workbench.WorkbenchUtils;

/**
 * UI handler to execute DB purge / Empty trash bin.
 * 
 * @author MarkoLuukkainen
 *
 */
public class DBPurgeHandler extends AbstractHandler {

	private boolean useRestart = true;
	private boolean preventOpenEditors = true;
	private boolean emptyTrashBin = true;
	
	@Override
	public Object execute(ExecutionEvent event) throws ExecutionException {
		String title = getTitle();
		if (preventOpenEditors) {
			IEditorPart activeEditor = HandlerUtil.getActiveEditor(event);
			if (activeEditor != null) {
				MessageDialog.open(MessageDialog.INFORMATION, HandlerUtil.getActiveShell(event), title, getEditorsOpenMessage(), 0);
				return null;
			}
			if (!WorkbenchUtils.closeEditors(HandlerUtil.getActiveWorkbenchWindow(event), true))
				return null;
		}
		String intro = getIntroMessage();
		if (intro != null) {
			if (!(MessageDialog.open(MessageDialog.QUESTION, HandlerUtil.getActiveShell(event), title, intro, 0)))
				return null;
		}
		
		IProgressService service = PlatformUI.getWorkbench().getProgressService();
		
		try {
			// TODO : While L0Utils.purgeDatabase takes ProgressMonitor as a parameter, it is not used. Additionally, purging is asynchronous process, and method call returns immediately. 
			service.busyCursorWhile(new IRunnableWithProgress() {
				@Override
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
					try {
						if (emptyTrashBin) {
							// Note : empty trash bin also purges the DB.
							Layer0Utils.emptyTrashBin(monitor);
						} else {
							Layer0Utils.purgeDatabase(monitor);
						}
					} catch (DatabaseException e) {
						throw new InvocationTargetException(e);
					}
				}
			});
		} catch (Throwable t) {
			ExceptionUtils.logAndShowError(getFailMessage(), t);
			return null;
		}
		// Since DB purge is async process, which cannot be monitored, we use db read to get notification of completion.
		Simantics.getSession().asyncRequest(new Read<Collection<Resource>>() {
			
			@Override
			public Collection<Resource> perform(ReadGraph graph) throws DatabaseException {
				Resource proj = Simantics.getProjectResource();
				Collection<Resource> models = readModels(graph, proj);
				return models;
				
			}
		}, new Procedure<Collection<Resource>>() {
			@Override
			public void execute(Collection<Resource> result) {
				if (useRestart) {
					Display.getDefault().asyncExec(new Runnable() {
						
						@Override
						public void run() {
							
							if (!MessageDialog.open(MessageDialog.QUESTION, HandlerUtil.getActiveShell(event), title, getRestartMessage(), 0))
								return;
							
							if (!PlatformUI.getWorkbench().restart()) {
								MessageDialog.open(MessageDialog.WARNING, HandlerUtil.getActiveShell(event), title, getRestartFailedMessage(), 0);
							}
						}
					});
				} else {Display.getDefault().asyncExec(new Runnable() {
						@Override
						public void run() {
							MessageDialog.open(MessageDialog.INFORMATION, HandlerUtil.getActiveShell(event), title, getCompletedMessage(), 0);
						}
					});
				}
			}
			
			@Override
			public void exception(Throwable throwable) {
				ExceptionUtils.logAndShowError(getFailModelsMessage(), throwable);
			}
		});

		return null;
	}
	
	public String getTitle() {
		return "DB Purge";
	}
	
	/**
	 * Returns intro message for the DB purge.
	 * 
	 * Returning null (if overridden) executes purge instantly without showing any messages.
	 * 
	 * @return
	 */
	public String getIntroMessage() {
		String msg = "DB purge is an experimental feature, which attempts to reduce size of the DB.\n\n"+
	             "Please meake sure that your workspace has a backup, or all models have been exported before running DB purge.\n\n"+
			     "Executing DB purge may lead to data loss!";
		return msg;
	}
	
	/**
	 * Shown if executing DB purge fails.
	 * 
	 * With current purge API implementation, this is never shown.
	 * 
	 * @return
	 */
	public String getFailMessage() {
		return "DB purge has failed.";
	}
	
	/**
	 * Message shown, if readModels query fails.
	 * @return
	 */
	public String getFailModelsMessage() {
		return "DB purge has failed, and we cannot retrieve models.";
	}
	
	/**
	 *  Shown after purge completion, and asking for application restart.
	 *  
	 * @return
	 */
	public String getRestartMessage() {
		String msg = "It is recommended to restart the application after DB purge.\n\n"+
			      "Restart the application?";
		return msg;
	}
	
	public String getRestartFailedMessage() {
		return "Application failed to restart.\n\n Please restart the application manually.";
	}
	
	/**
	 * Shown after purge completion.
	 * 
	 * Note: this message is shown only if useRestart=false.
	 * @return
	 */
	public String getCompletedMessage() {
		return "DB purge completed.";
	}
	
	/**
	 * Shown if there are open editors. 
	 * @return
	 */
	public String getEditorsOpenMessage() {
		return "Please close all editors before running DB purge.";
	}
	
	/**
	 * Database read operation to check if we can still read anything.
	 * 
	 * Please customize this for your application, if your software does not use Simulation ontology models. 
	 * 
	 * @param graph
	 * @param project
	 * @return
	 * @throws DatabaseException
	 */
	public Collection<Resource> readModels(ReadGraph graph, Resource project) throws DatabaseException {
		Layer0 L0 = Layer0.getInstance(graph);
		SimulationResource SIM = SimulationResource.getInstance(graph);
		Collection<Resource> models = graph.syncRequest(new ObjectsWithType(project, L0.ConsistsOf, SIM.Model));
		return models;
	}
	
	/**
	 * We recommended that Simantics is restarted after DB purge.
	 * 
	 * If situation changes, please change default value, or use setUseRestart.
	 * 
	 * @return true, if application is restarted automatically after DB purge. Default true.
	 */
	public boolean isUseRestart() {
		return useRestart;
	}
	
	public void setUseRestart(boolean useRestart) {
		this.useRestart = useRestart;
	}
	
	/**
	 * We recommended that all editors are closed, before executing DB purge.
	 * 
	 * If situation changes, please change default value, or use setPreventOpenEditors.
	 * 
	 * @return true, if running DB purge is prevented when an editor is open. Default true.
	 */
	public boolean isPreventOpenEditors() {
		return preventOpenEditors;
	}
	
	public void setPreventOpenEditors(boolean preventOpenEditors) {
		this.preventOpenEditors = preventOpenEditors;
	}
	
	public boolean isEmptyTrashBin() {
		return emptyTrashBin;
	}
	
	public void setEmptyTrashBin(boolean emptyTrashBin) {
		this.emptyTrashBin = emptyTrashBin;
	}

}
