/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench.editor;

import org.eclipse.jface.resource.ImageDescriptor;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.ui.workbench.ResourceEditorInput;
import org.simantics.utils.ui.ErrorLogger;

/**
 * A very simple and limited implementation of ResourceEditorAdapter.
 * 
 * <p>
 * Easy to use because all necessary information can be given as constructor
 * parameters.
 * </p>
 * 
 * <p>
 * Example:
 * 
 * <pre>
 * class MyEditorAdapter extends SimpleEditorAdapter {
 *     MyEditorAdapter() {
 *         super(&quot;My Editor&quot;, &quot;My.Editor.Id&quot;,
 *                 new String[] { Builtins.URIs.Entity },
 *                 new String[] { Builtins.URIs.Entity });
 *     }
 * }
 * </pre>
 * 
 * allows opening the editor "My.Editor.Id" for a resource if it is
 * inherited from the type Object or is an instance of Object.
 * </p>
 * 
 * @author Tuukka Lehtonen
 */
public class SimpleEditorAdapter extends AbstractResourceEditorAdapter {

    private final String   editorViewId;
    private final String   perspectiveId;
    private final String[] inherits;
    private final String[] instanceOf;

    /**
     * @param name
     * @param editorViewId
     * @param inherits
     * @param instanceOf
     */
    public SimpleEditorAdapter(String name, String editorViewId,
            String[] inherits, String[] instanceOf)
    {
        this(name, (ImageDescriptor) null, editorViewId, inherits, instanceOf);
    }

    /**
     * @param name
     * @param icon
     * @param editorViewId
     * @param inherits
     * @param instanceOf
     */
    public SimpleEditorAdapter(String name, ImageDescriptor icon,
            String editorViewId,
            String[] inherits, String[] instanceOf)
    {
        this(name, icon, editorViewId, null, inherits, instanceOf);
    }

    /**
     * @param name
     * @param icon
     * @param editorViewId
     * @param perspectiveId
     * @param inherits
     * @param instanceOf
     */
    public SimpleEditorAdapter(String name, ImageDescriptor icon,
            String editorViewId, String perspectiveId,
            String[] inherits, String[] instanceOf)
    {
        super(name, icon);
        this.editorViewId = editorViewId;
        this.perspectiveId = perspectiveId;
        this.inherits = inherits;
        this.instanceOf = instanceOf;
    }

    @Override
    public boolean canHandle(ReadGraph g, Resource r) throws DatabaseException {
        //System.out.println("canHandle: " + this.getClass().getCanonicalName());

        // If not filters are defined, allow anything.
        if ((inherits == null || inherits.length == 0)
                && (instanceOf == null || instanceOf.length == 0))
            return true;

        if (inherits != null) {
            for (String type : inherits) {
                //System.out.println("type: " + type);
                try {
                    if (g.isInheritedFrom(r, g.getResource(type)))
                        return true;
                } catch (DatabaseException e) {
                    ErrorLogger.defaultLogError("BUG: SimpleEditorAdapter " + getClass().getName()
                            + " checked for inheritance of '" + type + "' but resource was not found", e);
                }
            }
        }
        if (instanceOf != null) {
            for (String instance : instanceOf) {
                //System.out.println("instance: " + instance);
                try {
                    //System.out.println(instance);
                    if (g.isInstanceOf(r, g.getResource(instance)))
                        return true;
                } catch (DatabaseException e) {
                    ErrorLogger.defaultLogError("BUG: SimpleEditorAdapter " + getClass().getName()
                            + " checked for instanceOf '" + instance + "' but the type was not found", e);
                }
            }
        }
        return false;
    }


    @Override
    protected void openEditor(Resource r) throws Exception {
        assert(editorViewId != null);

        if (perspectiveId != null) {
            openEditorWithIdInPerspective(editorViewId, new ResourceEditorInput(editorViewId, r), perspectiveId);
        } else {
            openEditorWithId(editorViewId, new ResourceEditorInput(editorViewId, r));
        }
    }

    protected void openView(Resource r) throws Exception {
        assert(editorViewId != null);

        if (perspectiveId != null) {
            openViewWithIdInPerspective(editorViewId, r, perspectiveId);
        } else {
            openViewWithId(editorViewId, r);
        }
    }
    
    @Override
    public String getEditorId() {
    	return editorViewId;
    }

}
