package org.simantics.tests.modelled.utils;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.coverage.CombinedCoverage;
import org.simantics.scl.compiler.module.coverage.CoverageBuilder;
import org.simantics.utils.strings.AlphanumComparator;

public class ModelledSTSSuite {

    private String name;
    private List<ModelledSTSTest> children;
    private int priority;
    private String moduleNameFilter;
    private List<Pattern> moduleNameFilterPatterns = new ArrayList<>();

    private CoverageBuilder coverageBuilder;
    private Map<String, String> variables;

    ModelledSTSSuite(String name, List<ModelledSTSTest> children, String moduleNameFilter, int priority, Map<String, String> variables) {
        this.name = name;
        this.priority = priority;
        this.children = children;
        this.variables = variables;
        this.moduleNameFilter = moduleNameFilter;
        for (String s : moduleNameFilter.split(",")) {
            try {
                s = s.trim().replaceAll("\\*", "\\\\w*").toLowerCase();
                getModuleNameFilterPatterns().add(Pattern.compile(s));
            } catch (PatternSyntaxException e) {
                e.printStackTrace();
            }
        }
    }

    public String getName() {
        return name;
    }

    public List<ModelledSTSTest> getChildren() {
        return children;
    }

    static Comparator<ModelledSTSTest> comparator = (test1, test2) -> compareTests(test1, test2);
    
    public List<ModelledSTSTest> getSortedChildren() {
        Set<ModelledSTSTest> testsWithDeps = new HashSet<>();
        // This TreeMap sorts the tests with the comparator 
        TreeMap<ModelledSTSTest, String> sortedTests = new TreeMap<>(comparator);
        for (ModelledSTSTest test : getChildren()) {
            Set<String> testDependencies = test.getDependencies();
            if (testDependencies.isEmpty()) {
                // First tests that have no dependencies
                sortedTests.put(test, test.getName());
            } else {
                // These are resolved later
                testsWithDeps.add(test);
            }
        }
        
        // Construct a LinkedList that is returned as a result
        LinkedList<ModelledSTSTest> results = new LinkedList<>(sortedTests.keySet());
        
//        Set<ModelledSTSTest> temp = new HashSet<>(testsWithDeps);
        // Now resolve tests with dependencies
        for (ModelledSTSTest testWithDep : testsWithDeps) {
            boolean satisfied = true;
            for (String dep : testWithDep.getDependencies()) {
                if (!sortedTests.containsValue(dep)) {
                    satisfied = false;
                } else {
                    testWithDep.resolveDependency(dep);
                }
            }
            if (satisfied) {
                results.addLast(testWithDep);
                sortedTests.put(testWithDep, testWithDep.getName());
            } else {
                // Not satisfied
                System.out.println(testWithDep.getName() + " not satisfied");
            }
        }
        return results;
    }

    private static int compareTests(ModelledSTSTest test1, ModelledSTSTest test2) {
        if (test1.getPriority() < test2.getPriority())
            return -1;
        else if (test1.getPriority() > test2.getPriority())
            return 1;
        else
            return AlphanumComparator.COMPARATOR.compare(test1.getName(), test2.getName());
    }

    public String getModuleNameFilter() {
        return moduleNameFilter;
    }

    public void addCoverage(List<Module> modules) {
        if (coverageBuilder == null)
            coverageBuilder = new CoverageBuilder();
        for (Module module : modules)
            coverageBuilder.addCoverage(module, true);
    }

    public CombinedCoverage getCoverage() {
        if (coverageBuilder == null)
            return null;
        return coverageBuilder.getCoverage();
    }

    public List<Pattern> getModuleNameFilterPatterns() {
        return moduleNameFilterPatterns;
    }

    public int getPriority() {
        return priority;
    }

}
