package org.simantics.spreadsheet.graph.function;

import java.util.List;
import java.util.function.Consumer;

import org.simantics.databoard.binding.mutable.MutableVariant;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.request.Write;
import org.simantics.spreadsheet.CellEditor;
import org.simantics.spreadsheet.Range;
import org.simantics.spreadsheet.Transaction;
import org.simantics.spreadsheet.graph.SpreadsheetGraphUtils;
import org.simantics.spreadsheet.graph.celleditor.VariableCellEditorBase;
import org.simantics.spreadsheet.resource.SpreadsheetResource;
import org.simantics.spreadsheet.util.SpreadsheetUtils;

class SpreadsheetVariableCellEditor extends VariableCellEditorBase {

	public SpreadsheetVariableCellEditor(Variable spreadsheet) {
		super(spreadsheet);
	}

	@Override
	public void edit(final Transaction<Write> transaction, final String location, final String property, final Variant variant, Consumer<?> callback) {
		SpreadsheetUtils.schedule(transaction, new WriteRequest() {
			@Override
			public void perform(WriteGraph graph) throws DatabaseException {
				// Let's check the configuration - if needed the cell gets created into the sheet configuration
				CellEditor<Write> editor = getPossibleCellEditor(transaction, graph, location, variant);
				if (editor == null)
					return;
				editor.edit(transaction, location, property, variant, callback);
			}
		});
	}

	private CellEditor<Write> getPossibleCellEditor(final Transaction<Write> transaction, WriteGraph graph, String location, Variant value) throws DatabaseException {
		SpreadsheetResource SHEET = SpreadsheetResource.getInstance(graph);
		Range range = SpreadsheetUtils.decodePossibleCellAbsolute(location);
		if(range == null) return null; //No editor found

		List<Variable> cells = SpreadsheetGraphUtils.possibleConfigurationCellVariables(graph, configurationCell, range);
		if (cells.isEmpty()) {
			if (value == null) {
				return null;
			} else {
				cells = SpreadsheetGraphUtils.getOrCreateConfigurationCellVariables(graph, configurationCell, range);
				for(Variable cell : cells) {
					// Notify the transaction about the creation of the new cell
					transaction.needSynchronization(cell);
				}
			}
		}
		if (cells.size() != 1)
			throw new DatabaseException("Can edit only one cell at a time!");

		return cells.iterator().next().getPropertyValue(graph, SHEET.cellEditor);
	}

	@Override
	public void copy(final Transaction<Write> transaction, final String location, final MutableVariant variant, Consumer<?> callback) {

		SpreadsheetUtils.schedule(transaction, new WriteRequest() {

			@Override
			public void perform(WriteGraph graph) throws DatabaseException {

				SpreadsheetResource SHEET = SpreadsheetResource.getInstance(graph);
				Variable child = configurationCell.getPossibleChild(graph, location);
				if(child != null) {
					CellEditor<Write> editor = child.getPossiblePropertyValue(graph, SHEET.cellEditor);
					if(editor != null) {
						editor.copy(transaction, location, variant, null);
					}
				}
			}

		});

	}

}