package org.simantics.scl.compiler.compilation;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Externalizable;
import java.io.File;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectInputStream;
import java.io.ObjectOutput;
import java.io.ObjectOutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.simantics.scl.compiler.constants.Constant;
import org.simantics.scl.compiler.elaboration.java.ExternalConstantException;
import org.simantics.scl.compiler.elaboration.java.ExternalConstantReader;
import org.simantics.scl.compiler.elaboration.modules.SCLValue;
import org.simantics.scl.compiler.elaboration.modules.TypeClass;
import org.simantics.scl.compiler.elaboration.modules.TypeClassInstance;
import org.simantics.scl.compiler.elaboration.modules.TypeDescriptor;
import org.simantics.scl.compiler.internal.codegen.effects.EffectConstructor;
import org.simantics.scl.compiler.module.ConcreteModule;
import org.simantics.scl.compiler.module.ImportDeclaration;
import org.simantics.scl.compiler.module.options.ModuleCompilationOptions;
import org.simantics.scl.compiler.module.options.ModuleCompilationOptions.StoreOption;
import org.simantics.scl.compiler.module.options.StoredModuleSupport;
import org.simantics.scl.compiler.module.repository.ModuleRepository;
import org.simantics.scl.compiler.top.ModuleInitializer;
import org.simantics.scl.compiler.types.TCon;
import org.simantics.scl.compiler.types.Types;
import org.simantics.scl.runtime.SCLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StoredModule implements Externalizable  {
	
    private static final Logger LOGGER = LoggerFactory.getLogger(StoredModule.class);

	private static final long serialVersionUID = -1512783817837814358L;
	
	public String moduleName;
	public String classLoaderKey;
	
	public List<ImportDeclaration> dependencies;
	public Map<String,TypeDescriptor> typeDescriptors;
	public Map<String,TypeClass> typeClasses;
	public Map<TCon, ArrayList<TypeClassInstance>> typeClassInstances;
	public Map<String, EffectConstructor> effectConstructors;
	public Map<String, List<Constant>> fieldAccessors;
	public Collection<SCLValue> values;
	public ModuleInitializer moduleInitializer;
	public Map<String,byte[]> classes;
	
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("Classes\n");
		for(String name : classes.keySet()) {
			sb.append(name);
			sb.append(": ");
			sb.append(classes.get(name).length);
			sb.append("bytes\n");
		}
		return sb.toString();
	}
	
	public static StoredModule fromModule(StoredModuleSupport classLoaderProvider, ConcreteModule module) {
		StoredModule result = new StoredModule();
		result.moduleName = module.getName();
		result.classLoaderKey = classLoaderProvider.getClassLoaderKey(module.getParentClassLoader()); 
		result.dependencies = module.getDependencies();
		result.typeDescriptors = module.getTypeDescriptors();
		result.typeClasses = module.getTypeClassMap();
		result.typeClassInstances = module.getTypeInstances();
		result.effectConstructors = module.getEffectConstructors();
		result.values = module.getValues();
		result.fieldAccessors = module.getFieldAccessors(); 
		result.classes = module.getClasses();
		result.moduleInitializer = module.getModuleInitializer();
		return result;
	}
	
	public ConcreteModule toModule(StoredModuleSupport classLoaderProvider) {
		ConcreteModule module = new ConcreteModule(moduleName);
		module.setClasses(classes);
		module.setParentClassLoader(classLoaderProvider.getClassLoader(classLoaderKey));
		for(String name : typeClasses.keySet())
			module.addTypeClass(name, typeClasses.get(name));
		for(String name : typeDescriptors.keySet())
			module.addTypeDescriptor(name, typeDescriptors.get(name));
		for(TCon con : typeClassInstances.keySet()) {
			List<TypeClassInstance> list = typeClassInstances.get(con);
			for(TypeClassInstance tci : list)
				module.addTypeClassInstance(con, tci);
		}
		for(ImportDeclaration dependency : dependencies)
			module.addDependency(dependency);
		for(String name : effectConstructors.keySet())
			module.addEffectConstructor(name, effectConstructors.get(name));
		for(String name : fieldAccessors.keySet()) {
			List<Constant> list = fieldAccessors.get(name);
			for(Constant c : list)
				module.addFieldAccessor(name, c);
		}
		for(SCLValue value : values)
			module.addValue(value);
		module.setModuleInitializer(moduleInitializer);
		return module;
	}
	
	private static String storeFileName(String moduleName) {
        return moduleName.replace(':', '_').replace('\\', '_').replace('/', '_'); 
	}

	private static File storeFile(ModuleRepository repository, String moduleName) {

		Path path = repository.getLocation();
		if(path == null)
			return null;
		
		File parent = path.toFile();
		File scl = new File(parent, "scl");
		return new File(scl, storeFileName(moduleName));
		
	}
	
    public static ConcreteModule read(ModuleRepository repository, ModuleCompilationOptions options, String moduleName) {
    	
    	if(StoreOption.NONE.equals(options.store))
    		return null;
    	
    	try {

    		File fp = storeFile(repository, moduleName);
    		if(fp == null)
    			return null;
    		
    		if(!fp.exists())
    			return null;
    		
    		byte[] bytes = Files.readAllBytes(fp.toPath());
    		ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
    		ObjectInput oi = new ObjectInputStream(bais);
    		StoredModule cls = null;
			ExternalConstantReader old = (ExternalConstantReader)SCLContext.getCurrent().get(ExternalConstantReader.KEY);
    		try {
    			SCLContext.getCurrent().put(ExternalConstantReader.KEY, options.classLoaderProvider.getExternalConstantReader());
    		    cls = options.classLoaderProvider.readModule(oi);
    		    return cls.toModule(options.classLoaderProvider);
    		} finally {
    			SCLContext.getCurrent().put(ExternalConstantReader.KEY, old);
    		}
    	} catch (IOException e) {
    		LOGGER.error("Error while reading SCL module " + moduleName, e);
    	} catch (ClassNotFoundException e) {
    		LOGGER.error("Error while reading SCL module " + moduleName, e);
		}
    	
		return null;

    }

	public static void write(ModuleRepository repository, ModuleCompilationOptions options, ConcreteModule module) {
		
    	if(!StoreOption.WRITE.equals(options.store))
    		return;
    	
		if(!module.supportsStore())
			return;


    	try {
    		
    		File fp = storeFile(repository, module.getName());
    		if(fp == null)
    			return;

    		fp.getParentFile().mkdirs();
   		
    		StoredModule cls = StoredModule.fromModule(options.classLoaderProvider, module);
    		ByteArrayOutputStream baos = new ByteArrayOutputStream();
    		ObjectOutput oo = new ObjectOutputStream(baos);
			SCLContext.getCurrent().put(ExternalConstantReader.KEY, options.classLoaderProvider.getExternalConstantReader());
    		oo.writeObject(cls);
    		Files.write(fp.toPath(), baos.toByteArray(), 
    				StandardOpenOption.WRITE,
                    StandardOpenOption.TRUNCATE_EXISTING,
                    StandardOpenOption.CREATE);
    		
    		LOGGER.debug("Wrote " + module.getName());
    		
    	} catch (ExternalConstantException e) {
    		LOGGER.info("Module " + module.getName() + " was not written: " + e.getMessage());
    	} catch (IOException e) {
    		LOGGER.debug("Module " + module.getName() + " was not written: " + e.getMessage());
		}
    	
    }

	@Override
	public void writeExternal(ObjectOutput out) throws IOException {
		out.writeUTF(moduleName);
		out.writeUTF(classLoaderKey);
		out.writeObject(dependencies);
		out.writeInt(typeDescriptors.size());
		for(String key : typeDescriptors.keySet()) {
			TypeDescriptor td = typeDescriptors.get(key);
			out.writeUTF(key);
			out.writeObject(td);
		}
		out.writeInt(typeClasses.size());
		for(String key : typeClasses.keySet()) {
			TypeClass tc = typeClasses.get(key);
			out.writeUTF(key);
			out.writeObject(tc);
		}
		out.writeInt(typeClassInstances.size());
		for(TCon key : typeClassInstances.keySet()) {
			ArrayList<TypeClassInstance> list = typeClassInstances.get(key);
			out.writeObject(key);
			out.writeObject(list);
		}
		out.writeInt(effectConstructors.size());
		for(String key : effectConstructors.keySet()) {
			EffectConstructor ec = effectConstructors.get(key);
			out.writeUTF(key);
			out.writeObject(ec);
		}
		out.writeInt(fieldAccessors.size());
		for(String key : fieldAccessors.keySet()) {
			List<Constant> list = fieldAccessors.get(key);
			out.writeUTF(key);
			out.writeObject(list);
		}
		out.writeInt(values.size());
		for(SCLValue value : values)
			out.writeObject(value);
		out.writeObject(moduleInitializer);
		out.writeObject(classes);
	}

	@Override
	public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
		moduleName = in.readUTF();
		classLoaderKey = in.readUTF();
		dependencies = (List<ImportDeclaration>)in.readObject();
		{
			int len = in.readInt();
			typeDescriptors = new HashMap<>(len);
			for(int i=0;i<len;i++) {
				String name = in.readUTF();
				TypeDescriptor td = (TypeDescriptor)in.readObject();
				typeDescriptors.put(name, td);
			}
		}
		{
			int len = in.readInt();
			typeClasses = new HashMap<>(len);
			for(int i=0;i<len;i++) {
				String name = in.readUTF();
				TypeClass tc = (TypeClass)in.readObject();
				typeClasses.put(name, tc);
			}
		}
		{
			int len = in.readInt();
			typeClassInstances = new HashMap<>(len);
			for(int i=0;i<len;i++) {
				TCon con = Types.con((TCon)in.readObject());
				ArrayList<TypeClassInstance> list = (ArrayList<TypeClassInstance>)in.readObject();
				typeClassInstances.put(con, list);
			}
		}
		{
			int len = in.readInt();
			effectConstructors = new HashMap<>(len);
			for(int i=0;i<len;i++) {
				String name = in.readUTF();
				EffectConstructor ec = (EffectConstructor)in.readObject();
				effectConstructors.put(name, ec);
			}
		}
		{
			int len = in.readInt();
			fieldAccessors = new HashMap<>(len);
			for(int i=0;i<len;i++) {
				String key = in.readUTF();
				ArrayList<Constant> list = (ArrayList<Constant>)in.readObject();
				fieldAccessors.put(key, list);
			}
		}
		{
			int len = in.readInt();
			values = new ArrayList<>();
			for(int i=0;i<len;i++) {
				values.add((SCLValue)in.readObject());
			}
		}
		moduleInitializer = (ModuleInitializer)in.readObject();
		classes = (Map<String,byte[]>)in.readObject();
	}
	
}