package org.simantics.scl.compiler.commands;


import java.io.IOException;
import java.io.Reader;
import java.io.Writer;

import org.simantics.scl.compiler.errors.Failable;
import org.simantics.scl.compiler.errors.Failure;
import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.repository.ModuleRepository;
import org.simantics.scl.compiler.module.repository.UpdateListener;
import org.simantics.scl.compiler.source.StringModuleSource;
import org.simantics.scl.compiler.source.repository.MapModuleSourceRepository;
import org.simantics.scl.runtime.reporting.AbstractSCLReportingHandler;
import org.simantics.scl.runtime.reporting.SCLReportingHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommandSessionWithModules {
    private static final Logger LOGGER = LoggerFactory.getLogger(CommandSessionWithModules.class);
    
    private MapModuleSourceRepository localModuleSourceRepository;
    private ModuleRepository moduleRepository;
    private CommandSession commandSession;
    
    private static final SCLReportingHandler DEFAULT_REPORTING_HANDLER = new AbstractSCLReportingHandler() {
        @Override
        public void print(String text) {
            CommandSessionWithModules.LOGGER.info(text);
        }
    };
    
    public CommandSessionWithModules(ModuleRepository parentRepository) {
        this.localModuleSourceRepository = new MapModuleSourceRepository();
        this.moduleRepository = new ModuleRepository(parentRepository, localModuleSourceRepository);
        this.commandSession = new CommandSession(moduleRepository, DEFAULT_REPORTING_HANDLER);
        this.commandSession.setDependenciesListener(new UpdateListener() {
            @Override
            public void notifyAboutUpdate() {
                commandSession.updateRuntimeEnvironment(true);
            }
        });
    }
    
    public CommandSession getCommandSession() {
        return commandSession;
    }
    
    /**
     * Puts the given module to the local module repository. Returns null, if the
     * compilation of the module succeeded or a string containing the compilation
     * errors, if it failed.
     */
    public String putModule(String moduleName, String moduleText) {
        StringModuleSource moduleSource = new StringModuleSource(moduleName, moduleText);
        synchronized(localModuleSourceRepository) {
            localModuleSourceRepository.addModuleDescriptor(moduleSource);
        }
        Failable<Module> module = moduleRepository.getModule(moduleName);
        if(module.didSucceed())
            return null;
        else
            return ((Failure)module).toString(moduleText);
    }
    
    /**
     * Runs commands read from commandReader and writes responses to
     * responseWriter.
     */
    public void runCommands(Reader commandReader, Writer responseWriter) {
        SCLReportingHandler handler = new AbstractSCLReportingHandler() {
            @Override
            public void printCommand(String command) {
                // Don't echo commands
            }
            @Override
            public void print(String text) {
                try {
                    responseWriter.write(text + "\n");
                    responseWriter.flush();
                } catch (IOException e) {
                    CommandSessionWithModules.LOGGER.error("Writing reponse failed.", e);
                }
            }
        };
        commandSession.execute(commandReader, handler);
    }
}
