/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.scenegraph.g2d.nodes;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;

import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.scenegraph.utils.TextUtil;

public class GalleryItemNode extends TransformNode {

    private static final long serialVersionUID = -423827087534629381L;

    protected String text = "";
    protected Rectangle2D bounds = null;
    protected Rectangle2D imagebounds = null;
    protected Font font = null;

    @SyncField({"text", "bounds", "font", "imagebounds"})
    public void init(String text, Rectangle2D bounds, Font font, Rectangle2D imagebounds) {
        this.text = text;
        this.bounds = bounds;
        this.font = font;
        this.imagebounds = imagebounds;
    }

    @Override
    public void render(Graphics2D g2d) {
        double contentDim = Math.min(bounds.getWidth(), bounds.getHeight());

        AffineTransform ot = g2d.getTransform();

        if(!transform.isIdentity()) {
            g2d.transform(transform);
        }

        if (text != null) {
            Graphics2D g = (Graphics2D) g2d.create();

            g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
            //Rectangle2D textRectangle = new Rectangle2D.Double(0,0, 0, bounds.getHeight()-contentDim);
            Rectangle2D textRectangle = new Rectangle2D.Double(0, contentDim, bounds.getWidth(), bounds.getHeight()-contentDim);
            g.setClip(textRectangle);
            g.setColor(Color.BLACK);

            g.setFont(font); // FIXME
            FontMetrics metrics = g.getFontMetrics();

            int fontHeight = metrics.getHeight();
            float width = (float) bounds.getWidth();
            float y = (float) bounds.getWidth() + fontHeight;
            int i = 0;
            for (String s : TextUtil.wordWrap(text, metrics, (int) textRectangle.getWidth())) {
                int sWidth = metrics.stringWidth(s);
                g.drawString(s, (width - sWidth) / 2,  y + i * fontHeight);
                ++i;
            }

            g.dispose();
        }

        for(IG2DNode n : getSortedNodes()) {
            double imgMaxDim = Math.max(imagebounds.getWidth(), imagebounds.getHeight());
            double scaleFactor = contentDim / imgMaxDim;
            // Offsets for centering the image and the shadow
            double xCenteringOffset = -(imagebounds.getWidth() - imgMaxDim) / 2;
            double yCenteringOffset = -(imagebounds.getHeight() - imgMaxDim) / 2;

            AffineTransform at = g2d.getTransform();
            g2d.scale(scaleFactor, scaleFactor);
            g2d.translate(-imagebounds.getMinX(), -imagebounds.getMinY());
            g2d.translate(xCenteringOffset, yCenteringOffset);
            n.render(g2d);
            g2d.setTransform(at);
        }

        g2d.setTransform(ot);
    }

    @Override
    public Rectangle2D getBoundsInLocal() {
        return bounds;
    }

    @Override
    public String toString() {
        return super.toString() + " [text=" + text + "]";
    }

}
