/*******************************************************************************
 * Copyright (c) 2012, 2013 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.internal;

import gnu.trove.map.hash.THashMap;

import java.util.Collections;
import java.util.Set;

import org.simantics.db.WriteGraph;
import org.simantics.objmap.backward.IBackwardLinkType;
import org.simantics.objmap.backward.IBackwardMapping;
import org.simantics.objmap.backward.IBackwardMappingSchema;
import org.simantics.objmap.exceptions.MappingException;

/**
 * An unidirectional (from range to domain) mapping that does not support removals.
 * 
 * @author Hannu Niemist&ouml;
 */
public class MonotoneBackwardMapping<Domain, Range> implements IBackwardMapping<Domain, Range> {

    IBackwardMappingSchema<Domain, Range> schema;
    THashMap<Range, Domain> map = new THashMap<Range, Domain>();
    
    public MonotoneBackwardMapping(IBackwardMappingSchema<Domain, Range> schema) {
        this.schema = schema;
    }

    @Override
    public Set<Range> getRange() {
        return Collections.unmodifiableSet(map.keySet());
    }

    @Override
    public Domain inverseGet(Range rangeElement) {
        return map.get(rangeElement);
    }

    @Override
    public Domain inverseMap(WriteGraph graph, Range rangeElement) throws MappingException {
        Domain result = inverseGet(rangeElement);
        if(result == null) {
            IBackwardLinkType<Domain, Range> linkType = 
                    schema.linkTypeOfRangeElement(graph, rangeElement);
            // Two phase creation makes cyclic references possible
            Domain domainElement = linkType.createDomainElement(graph, rangeElement);
            map.put(rangeElement, domainElement);
            linkType.createDomain(graph, this, domainElement, rangeElement);
        }
        return result;
    }

}
