/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.graph.impl;


import java.util.AbstractSet;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.simantics.db.ReadGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.objmap.backward.IBackwardMapping;
import org.simantics.objmap.exceptions.MappingException;
import org.simantics.objmap.forward.IForwardMapping;
import org.simantics.objmap.graph.IMapping;
import org.simantics.objmap.graph.IMappingListener;
import org.simantics.objmap.graph.schema.ILinkType;
import org.simantics.objmap.graph.schema.IMappingSchema;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gnu.trove.map.hash.THashMap;

/**
 * An implementation of IMapping. The class should not be created
 * directly but using methods in Mappings.
 * @see org.simantics.objmap.graph.Mappings
 * @author Hannu Niemist
 */
public class Mapping<Domain, Range> implements IMapping<Domain, Range> {

	static final Logger LOGGER = LoggerFactory.getLogger(Mapping.class);
	
	
	IMappingSchema<Domain, Range> schema;
	
	THashMap<Domain, Link<Domain,Range>> domain = new THashMap<Domain, Link<Domain,Range>>();
	THashMap<Range, Link<Domain,Range>> range = new THashMap<Range, Link<Domain,Range>>();
	ArrayList<IMappingListener> listeners = new ArrayList<IMappingListener>();

	ArrayList<Link<Domain,Range>> modifiedDomainLinks = new ArrayList<Link<Domain,Range>>();
	ArrayList<Link<Domain,Range>> modifiedRangeLinks = new ArrayList<Link<Domain,Range>>();

	boolean disposed = false;
	
	boolean listensDomain; 
	
	public Mapping(IMappingSchema<Domain, Range> schema, boolean listensDomain) {
		this.schema = schema;
		this.listensDomain = listensDomain;
	}
	
	private void removeLink(Link<Domain,Range> link) {
		if(link.domainModified)
			modifiedDomainLinks.remove(link);
		if(link.rangeModified)
			modifiedRangeLinks.remove(link);
		link.removed = true;
	}
	
	private void createDomain(WriteGraph g, Link<Domain,Range> link) throws MappingException {
	    LOGGER.trace("        createDomain for " + link.rangeElement);
		ILinkType<Domain,Range> type = schema.linkTypeOfRangeElement(link.rangeElement);
		Domain domainElement = type.createDomainElement(g, link.rangeElement);
		link.type = type;
		link.domainElement = domainElement;
		domain.put(domainElement, link);
		type.createDomain(g, new RangeToDomain(), domainElement, link.rangeElement);
		
        // TODO Should we do this only if the mapping is listening?
        domainModified(link);
	}
	
	private void createRange(ReadGraph g, Link<Domain,Range> link) throws MappingException {
		ILinkType<Domain,Range> type = schema.linkTypeOfDomainElement(g, link.domainElement);		
		Range rangeElement = type.createRangeElement(g, link.domainElement);
		
		link.type = type;
		link.rangeElement = rangeElement;
		range.put(rangeElement, link);
		type.createRange(g, new DomainToRange(), link.domainElement, rangeElement);
	}
	
	Set<Domain> domainSet = new AbstractSet<Domain>() {

		public boolean add(Domain e) {
			if(domain.containsKey(e))
				return false;
			Link<Domain,Range> link = new Link<Domain,Range>(null, e, null);
			domain.put(e, link);
			modifiedDomainLinks.add(link);
			return true;
		}
		
		public boolean contains(Object o) {
			return domain.contains(o);
		}
		
		public boolean remove(Object o) {
			Link<Domain,Range> link = domain.remove(o);			
			if(link == null)
				return false;
			removeLink(link);
			if(link.rangeElement != null)
				range.remove(link.rangeElement);
			return true;	
		}
		
		@Override
		public Iterator<Domain> iterator() {
		    // FIXME does not implement Iterator.remove correctly
			return domain.keySet().iterator();
		}

		@Override
		public int size() {
			return domain.size();
		}
		
	};
	
	Set<Range> rangeSet = new AbstractSet<Range>() {

		public boolean add(Range e) {
			if(range.containsKey(e))
				return false;
			Link<Domain,Range> link = new Link<Domain,Range>(null, null, e);
			range.put(e, link);
			modifiedRangeLinks.add(link);
			return true;
		}
		
		public boolean contains(Object o) {
			return range.contains(o);
		}
		
		public boolean remove(Object o) {
			Link<Domain,Range> link = range.remove(o);			
			if(link == null)
				return false;
			removeLink(link);
			if(link.domainElement != null)
				domain.remove(link.domainElement);
			return true;
		}
		
		@Override
		public Iterator<Range> iterator() {
		    // FIXME does not implement Iterator.remove correctly
			return range.keySet().iterator();
		}

		@Override
		public int size() {
			return range.size();
		}
		
	};
	
	class DomainToRange implements IForwardMapping<Domain, Range> {

		public DomainToRange() {
		}

		@Override
		public Range get(Domain element)  {
			Link<Domain,Range> link = domain.get(element);
			if (link != null)
				return link.rangeElement;
			return null;
			
		}
		
		@Override
		public Range map(ReadGraph graph, Domain element)
				throws MappingException {
			Link<Domain,Range> link = domain.get(element);
			if(link == null) {
			    link = new Link<Domain,Range>(null, element, null);
	            link.domainModified = true;
	            modifiedDomainLinks.add(link);
			    domain.put(element, link);           
			    createRange(graph, link);	
			}
			else if(link.type == null) 
				createRange(graph, link);
            return link.rangeElement;
		}
		
		@Override
		public Set<Domain> getDomain() {
			return domain.keySet();
		}
		
	};
	
	class RangeToDomain extends DomainToRange implements IBackwardMapping<Domain, Range> {

		public RangeToDomain() {
			super();
		}
		
		@Override
		public Domain inverseGet(Range element) {
			
			Link<Domain,Range> link = range.get(element);
			if(link != null)
				return link.domainElement;
			return null;
		}
		
		@Override
		public Domain inverseMap(WriteGraph graph, Range element)
				throws MappingException {
			Link<Domain,Range> link = range.get(element);
			if(link == null) {
			    link = new Link<Domain,Range>(null, null, element);
			    link.rangeModified = true;
                modifiedRangeLinks.add(link);
			    range.put(element, link);
			    createDomain(graph, link);				
			}
			else if(link.type == null)
				createDomain(graph, link);
			return link.domainElement;
		}
		
		
		@Override
		public Set<Range> getRange() {
			return range.keySet();
		}
	};
	
	@Override
	public Set<Domain> getDomain() {
		return domainSet;
	}
	
	@Override
	public Set<Range> getRange() {
		return rangeSet;
	}
	
	
	@Override
	public synchronized Collection<Domain> updateDomain(WriteGraph g) throws MappingException {
	    LOGGER.trace("Mapping.updateDomain");
		RangeToDomain map = new RangeToDomain();
		ArrayList<Domain> updated = new ArrayList<Domain>();
		while(!modifiedRangeLinks.isEmpty()) {
		    LOGGER.trace("    modifiedRangeLinks.size() = " + modifiedRangeLinks.size());
		    
			Link<Domain,Range> link = modifiedRangeLinks.remove(modifiedRangeLinks.size()-1);
			link.rangeModified = false;
			/*if(link.domainModified) {
				link.domainModified = false;
				modifiedDomainLinks.remove(link);
			}*/
			
			if(link.type == null) {
				createDomain(g, link);
			}
			else {
				if(link.type.updateDomain(g, map, link.domainElement, link.rangeElement))
					updated.add(link.domainElement);
			}
		}	
		if (listensDomain)
			updateRange(g); //FIXME: without this listening would stop. 
		return updated;
	}
	
	@Override
	public synchronized Collection<Range> updateRange(ReadGraph g) throws MappingException {
	    LOGGER.trace("Mapping.updateRange");
		DomainToRange map = new DomainToRange();
		ArrayList<Range> updated = new ArrayList<Range>();
		while(!modifiedDomainLinks.isEmpty()) {		    
		    LOGGER.trace("    modifiedDomainLinks.size() = " + modifiedDomainLinks.size());
		    
			Link<Domain,Range> link = modifiedDomainLinks.remove(modifiedDomainLinks.size()-1);
			link.domainModified = false;
			/*if(link.rangeModified) {
				link.rangeModified = false;
				modifiedRangeLinks.remove(link);
			}*/
			
			if(link.type == null) {
				createRange(g, link);
			}
			
			if(listensDomain) {
			    RangeUpdateRequest<Domain,Range> request = new RangeUpdateRequest<Domain,Range>(link, map, this);
			    boolean changes;
			    try {
			    	changes = g.syncRequest(request, request) > 0;
                } catch (DatabaseException e) {
                    throw new MappingException(e);
                }
			    
			    if (changes)
			    	updated.add(link.rangeElement);
			}
			else
			    if(link.type.updateRange(g, map, link.domainElement, link.rangeElement))
			    	updated.add(link.rangeElement);
		}	
		return updated;
	}

	@Override
	public Range get(Domain domainElement) {
		Link<Domain,Range> link = domain.get(domainElement);
		if(link == null)
			return null;
		return link.rangeElement;
	}

	@Override
	public Domain inverseGet(Range rangeElement) {
		Link<Domain,Range> link = range.get(rangeElement);
		if(link == null)
			return null;
		return link.domainElement;
	}

	@Override
	public Domain inverseMap(WriteGraph g, Range rangeElement) throws MappingException {
		getRange().add(rangeElement);
		updateDomain(g);
		return inverseGet(rangeElement);
	}

	@Override
	public Range map(ReadGraph g, Domain domainElement) throws MappingException {
		getDomain().add(domainElement);
		updateRange(g);
		return get(domainElement);
	}

	void domainModified(Link<Domain,Range> link) {
	    if(!link.domainModified) {	        
	        synchronized(modifiedDomainLinks) {
	            LOGGER.trace("        domainModified for " + link.rangeElement);
                link.domainModified = true;
                modifiedDomainLinks.add(link);
                if(modifiedDomainLinks.size() == 1) {
                    for(IMappingListener listener : listeners)
                        listener.domainModified();
                }
	        }
        }
	}
	
	@Override
	public void domainModified(Domain domainElement) {
		Link<Domain,Range> link = domain.get(domainElement);
		if(link != null)
		    domainModified(link);
	}

	void rangeModified(Link<Domain,Range> link) {
	    if(!link.rangeModified) {
	        synchronized(modifiedRangeLinks) {
                link.rangeModified = true;
                modifiedRangeLinks.add(link);
                if(modifiedRangeLinks.size() == 1) {
                    for(IMappingListener listener : listeners)
                        listener.rangeModified();
                }
	        }
        }
	}
	
	@Override
	public void rangeModified(Range rangeElement) {
		Link<Domain,Range> link = range.get(rangeElement);
		if(link != null)
		    rangeModified(link);
	}

	@Override
	public boolean isDomainModified() {
		return !modifiedDomainLinks.isEmpty();
	}

	@Override
	public boolean isRangeModified() {
		return !modifiedRangeLinks.isEmpty();
	}
	
	@Override
	public Collection<Domain> getDomainModified() {
		List<Domain> list = new ArrayList<Domain>(modifiedDomainLinks.size());
		for (Link<Domain, Range> link : modifiedDomainLinks)
			list.add(link.domainElement);
		return list;
				
	}
	
	@Override
	public Collection<Range> getRangeModified() {
		List<Range> list = new ArrayList<Range>(modifiedRangeLinks.size());
		for (Link<Domain, Range> link : modifiedRangeLinks)
			list.add(link.rangeElement);
		return list;
	}

	@Override
	public void addMappingListener(IMappingListener listener) {
		listeners.add(listener);
	}

	@Override
	public void removeMappingListener(IMappingListener listener) {
		listeners.remove(listener);		
	}

	@Override
	public Collection<Domain> getConflictingDomainElements() {
		ArrayList<Domain> result = new ArrayList<Domain>();
		if(modifiedDomainLinks.size() < modifiedRangeLinks.size()) {
			for(Link<Domain,Range> link : modifiedDomainLinks)
				if(link.rangeModified)
					result.add(link.domainElement);
		}
		else {
			for(Link<Domain,Range> link : modifiedRangeLinks)
				if(link.domainModified)
					result.add(link.domainElement);
		}
		return result;
	}

	@Override
	public Collection<Range> getConflictingRangeElements() {
		ArrayList<Range> result = new ArrayList<Range>();
		if(modifiedDomainLinks.size() < modifiedRangeLinks.size()) {
			for(Link<Domain,Range> link : modifiedDomainLinks)
				if(link.rangeModified)
					result.add(link.rangeElement);
		}
		else {
			for(Link<Domain,Range> link : modifiedRangeLinks)
				if(link.domainModified)
					result.add(link.rangeElement);
		}
		return result;
	}

    @Override
    public void dispose() {
        disposed = true;
    }
    
    public boolean isDisposed() {
        return disposed;
    }
	
}
