/*******************************************************************************
 *  Copyright (c) 2023 Association for Decentralized Information Management
 *  in Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.web;

import java.awt.RenderingHints;
import java.awt.geom.AffineTransform;
import java.io.StringWriter;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.batik.dom.GenericDOMImplementation;
import org.apache.batik.svggen.SVGGraphics2D;
import org.apache.batik.svggen.SVGGraphics2DIOException;
import org.apache.batik.svggen.SVGIDGenerator;
import org.simantics.diagram.elements.TextNode;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.participant.ElementPainter.SelectionShapeNode;
import org.simantics.g2d.element.IElement;
import org.simantics.scenegraph.INode;
import org.simantics.scenegraph.g2d.G2DParentNode;
import org.simantics.scenegraph.g2d.G2DRenderingHints;
import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.scenegraph.g2d.IG2DNodeVisitor;
import org.simantics.scenegraph.g2d.nodes.SVGNode;
import org.simantics.scenegraph.g2d.nodes.SelectionNode;
import org.simantics.scenegraph.g2d.nodes.connection.RouteGraphNode;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * @author Jussi Koskela
 * @since 1.57.0
 */
public class JSONGenerator implements IG2DNodeVisitor {

	private static final boolean RENDER_ROUTEGRAPH_CLIENTSIDE = true;

	private SVGIDGenerator idGenerator = new SVGIDGenerator();
	private List<ObjectNode> nodes = new ArrayList<ObjectNode>();
	private ObjectMapper mapper;
	//private Map<String, String> defs = new HashMap<String, String>(); // svg data to def id
	private Map<String, String> newDefs = new HashMap<String, String>(); // def id to svg data
	private IDiagram diagram;
	private int elements = 0;

	public Map<String, String> flushNewDefs() {
		Map<String, String> result = newDefs;
		newDefs = new HashMap<String, String>();
		return result;
	}

	public JSONGenerator(ObjectMapper mapper, IDiagram diagram) {
		this.mapper = mapper;
		this.diagram = diagram;
	}

	public List<ObjectNode> getNodes() {
		return nodes;
	}


	@Override
	public void enter(IG2DNode node) {
		Map<INode, IElement> nodeToElement = diagram.getHint(DiagramHints.NODE_TO_ELEMENT_MAP);
		IElement e = nodeToElement.get(node);
		if (e != null) elements++;
		if (elements == 1) {
			node.refresh();
			ObjectNode n = createJsonNode(node);
			if (n != null) {
				nodes.add(n);
			}
		}
	}

	@Override
	public void leave(IG2DNode node) {
		Map<INode, IElement> nodeToElement = diagram.getHint(DiagramHints.NODE_TO_ELEMENT_MAP);
		IElement e = nodeToElement.get(node);
		if (e != null) elements--;
	}

	public String asSVG(IG2DNode node, boolean identityTransform) {
		// Get a DOMImplementation.
		DOMImplementation domImpl = GenericDOMImplementation.getDOMImplementation();

		// Create an instance of org.w3c.dom.Document.
		String svgNS = "http://www.w3.org/2000/svg";
		Document document = domImpl.createDocument(svgNS, "svg", null);

		// Create an instance of the SVG Generator.
		SVGGraphics2D svgGenerator = new SVGGraphics2D(document);
		svgGenerator.getGeneratorContext().setIDGenerator(idGenerator);

		svgGenerator.setRenderingHint(G2DRenderingHints.KEY_TEXT_RENDERING_MODE, G2DRenderingHints.TextRenderingMode.AS_TEXT);

		svgGenerator.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		svgGenerator.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
		svgGenerator.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
		svgGenerator.setRenderingHint(RenderingHints.KEY_COLOR_RENDERING, RenderingHints.VALUE_COLOR_RENDER_QUALITY);
		svgGenerator.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);

		if (identityTransform) {
			AffineTransform save = node.getTransform();
			node.setTransform(AffineTransform.getTranslateInstance(0, 0)); // set to identity
			node.render(svgGenerator);
			node.setTransform(save);   
		} else {
			node.render(svgGenerator);
		}
		return printSVGDocument(svgGenerator);
	}

	public SVGNode asSVGNode(IG2DNode node) {
		SVGNode svgNode = new SVGNode();

		svgNode.setData(asSVG(node, true));
		svgNode.setTransform(node.getTransform());

		return svgNode;
	}

	public ObjectNode createJsonNode(IG2DNode node) {
		Long overrideId = null;

		INode parent = node.getParent();        

		// Disable SelectionSode rendering
		if (node instanceof SelectionNode || node instanceof SelectionShapeNode) {
			return null;
		}
		// Disable selection rendering for RouteGraphNode 
		if (node instanceof RouteGraphNode) {
			((RouteGraphNode)node).setIgnoreSelection(true);
		}

		if (!canRender(node)) {
			overrideId = node.getId();
			node = asSVGNode(node);
		}

		ObjectNode n = toJSON(mapper, node);

		if (overrideId != null) {
			n.put("id", overrideId.longValue());
		}

		if (parent != null) {
			n.put("parent", parent.getId());
		}

		if (node instanceof SVGNode) {
			/*String data = ((SVGNode)node).getData();
            String defsId = defs.get(data);
            if (defsId == null) {
                defsId = "S" + Integer.toString(defs.size());
                defs.put(data, defsId);
                newDefs.put(defsId, data);
            }
            n.put("defaultData", "<use xlink:href=\"#" + defsId + "\">");*/

			SVGNode svgNode = (SVGNode)node;
			JsonNode dd = n.remove("data");
			if(svgNode.hasAssignments()) {
				n.replace("defaultData", new com.fasterxml.jackson.databind.node.TextNode(svgNode.getDynamicSVG()));
			}

		}


		return n;
	}

	public static ObjectNode toJSON(ObjectMapper mapper, Object object) {
		ObjectNode n = new ObjectNode(JsonNodeFactory.instance);
		ArrayNode types = n.putArray("types");

		Class<?> clazz = object.getClass();

		while (clazz != null && clazz != Object.class) {
			types.add(clazz.getCanonicalName());
			for (Field field : clazz.getDeclaredFields()) {
				// Ignore transient properties, those are generally just caches.
				if (Modifier.isTransient(field.getModifiers()))
					continue;

				// Ignore statics, those shouldn't affect data transfer.
				if (Modifier.isStatic(field.getModifiers()))
					continue;

				boolean accessible = field.isAccessible();
				try {
					if (!accessible)
						field.setAccessible(true);
					Object value = field.get(object);

					//System.out.println(field.getName());
					n.set(field.getName(), mapper.valueToTree(value));

				} catch (IllegalArgumentException e) {
					e.printStackTrace();
				} catch (IllegalAccessException e) {
					e.printStackTrace();
				} finally {
					if (!accessible)
						field.setAccessible(false);
				}
			}
			clazz = clazz.getSuperclass();
		}
		return n;
	}

	public static String printSVGDocument(SVGGraphics2D svgGenerator) {

		StringWriter writer = new StringWriter();
		try {
			svgGenerator.stream(svgGenerator.getRoot(), writer, false, false);
		} catch (SVGGraphics2DIOException e) {
			//e.printStackTrace();
		}
		return writer.toString();

	}    

	private static boolean canRender(IG2DNode node) {
		return (node instanceof G2DParentNode || node instanceof SVGNode || node instanceof TextNode || (node instanceof RouteGraphNode && RENDER_ROUTEGRAPH_CLIENTSIDE));
	}
}