/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.modeling.ui.actions;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.jface.action.Action;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.common.NodeContextBuilder;
import org.simantics.browsing.ui.model.actions.ActionBrowseContext;
import org.simantics.browsing.ui.model.actions.IActionCategory;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.project.ontology.ProjectResource;
import org.simantics.ui.DoubleClickEvent;
import org.simantics.ui.IDoubleClickAction;
import org.simantics.ui.utils.ResourceAdaptionUtils;
import org.simantics.utils.ui.action.PriorityAction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ModeledDoubleClickActions implements IDoubleClickAction, IExecutableExtension {

    private static final Logger LOGGER = LoggerFactory.getLogger(ModeledDoubleClickActions.class);

    private Set<String> browseContexts = defaultBrowseContexts;

    final static public Set<String> defaultBrowseContexts = Collections.singleton(ProjectResource.URIs.ProjectDoubleClickActionContext);
    
    public void setInitializationData(IConfigurationElement config, String propertyName, Object data) throws CoreException {
    	if(data instanceof String) {
    		String str = (String)data;
    		String[] parms = str.split(";"); //$NON-NLS-1$
    		for(String parm : parms) {
    			String[] keyValue = parm.split("="); //$NON-NLS-1$
    			if(keyValue.length == 2) {
    				String key = keyValue[0].trim();
    				if("context".equals(key)) { //$NON-NLS-1$
    					browseContexts = Collections.singleton(keyValue[1]);
    				}
    			}
    		}
    	}
    }
    
    public Collection<String> getBrowseContexts() {
        return browseContexts;
    }
    
    public Collection<Resource> getBrowseContextResources(ReadGraph graph) throws DatabaseException {
        Collection<String> names = getBrowseContexts(); 
        ArrayList<Resource> result = new ArrayList<Resource>(names.size());
        for(String name : names)
            result.add(graph.getResource(name));
        return result;
    }

    @Override
    public void doubleClickEvent(DoubleClickEvent e) throws DatabaseException {

        ReadGraph g = e.getGraph();
        final Resource resource = ResourceAdaptionUtils.toSingleResource(e.getResource());
        if (resource == null)
            return;

        Collection<PriorityAction> actions = g.syncRequest(new ResourceRead<Collection<PriorityAction>>(resource) {

			@Override
			public Collection<PriorityAction> perform(ReadGraph graph) throws DatabaseException {
				
		    	List<NodeContext> contexts = Collections.singletonList(NodeContextBuilder.buildWithInput(resource));
		    	
		        try {

		        	NodeContext nodeContext = contexts.get(0);
		        	
		            ActionBrowseContext defaultABC = ActionBrowseContext.create(graph,
		                    getBrowseContextResources(graph));

		            ActionBrowseContext nodeABC = ActionBrowseContext.resolve(graph, nodeContext);
		            
	                ActionBrowseContext browseContext = defaultABC.merge(graph, nodeABC);
		            
		            Map<IActionCategory, List<Action>> result = browseContext.getActions(graph, nodeContext, contexts);

		            
		            return toContributionItems(result);
		            
		        } catch (DatabaseException e) {
		        	
		        	LOGGER.error("Error while resolving modeled double-click actions", e);
		        	
		        }
		        
		        return Collections.emptyList();
		        
			}
        	
        });
        
        for(PriorityAction act : actions)
        	e.add(act);
        
    }
    
    private static Collection<PriorityAction> toContributionItems(
            Map<IActionCategory, List<Action>> map) {

    	if(map.isEmpty())
            return Collections.emptyList();
        
        IActionCategory[] categories = map.keySet().toArray(new IActionCategory[map.size()]);
        Arrays.sort(categories, IActionCategory.ACTION_CATEGORY_COMPARATOR);
        
        ArrayList<PriorityAction> items = new ArrayList<PriorityAction>();
        for(IActionCategory category : categories) {
            List<Action> actions = map.get(category);
            for(Action act : actions)
                items.add(new PriorityAction(PriorityAction.HIGH+10) {
                    @Override
                    public void run() {
                    	act.run();
                    }
                });
        }
        return items;
    }    

}
