package org.simantics.modeling.template2d;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.container.DataContainer;
import org.simantics.databoard.container.DataContainers;
import org.simantics.databoard.container.DataFormatException;
import org.simantics.databoard.container.FormatHandler;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.impl.DefaultPasteImportAdvisor;
import org.simantics.graph.db.TGStatusMonitor;
import org.simantics.graph.db.TransferableGraphs;
import org.simantics.graph.representation.TransferableGraph1;

/**
 * Headless utility facade for diagram template I/O.
 * 
 * @author Tuukka Lehtonen
 */
public class DiagramTemplates {

    /**
     * @param monitor
     * @param session the database session to use for importing
     * @param templateFile
     *            the template file to import
     * @param targetContainer
     *            the database container resource to make the imported template
     *            a part of
     * @return the root resource describing imported template
     * @throws IOException
     * @throws DatabaseException
     */
    public static Resource importTemplate(
            IProgressMonitor monitor,
            final Session session,
            File templateFile,
            Resource targetContainer)
                    throws IOException, DatabaseException
    {
        final SubMonitor mon = SubMonitor.convert(monitor, "Importing diagram template", 100);
        final DefaultPasteImportAdvisor advisor = new DefaultPasteImportAdvisor(targetContainer);
        FormatHandler<Resource> handler = new DiagramTemplateFormatHandler() {
            @Override
            public Resource process(DataContainer container) throws Exception {
                TransferableGraphs.importGraph1WithMonitor(
                        session,
                        (TransferableGraph1)container.content.getValue(),
                        advisor,
                        new TGMonitor(mon));
                return advisor.getRoot();
            }
        };
        return importTemplate(templateFile, targetContainer, handler);
    }

    /**
     * @param monitor
     * @param graph
     *            database transaction handle to use for importing
     * @param templateFile
     *            the template file to import
     * @param targetContainer
     *            the database container resource to make the imported template
     *            a part of
     * @return the root resource describing imported template
     * @throws IOException
     * @throws DatabaseException
     */
    public static Resource importTemplate(
            IProgressMonitor monitor,
            final WriteGraph graph,
            File templateFile,
            Resource targetContainer)
                    throws IOException, DatabaseException
    {
        final SubMonitor mon = SubMonitor.convert(monitor, "Importing diagram template", 100);
        final DefaultPasteImportAdvisor advisor = new DefaultPasteImportAdvisor(targetContainer);
        FormatHandler<Resource> handler = new DiagramTemplateFormatHandler() {
            @Override
            public Resource process(DataContainer container) throws Exception {
                TransferableGraphs.importGraph1(graph,
                        (TransferableGraph1) container.content.getValue(),
                        advisor,
                        new TGMonitor(mon));
                return advisor.getRoot();
            }
        };
        return importTemplate(templateFile, targetContainer, handler);
    }

    private static Resource importTemplate(
            File modelFile,
            Resource target,
            FormatHandler<Resource> handler)
                    throws IOException, DatabaseException
    {
        try {
            Map<String, FormatHandler<Resource>> handlers = new HashMap<>(2);
            handlers.put(DiagramTemplateConstants.DRAWING_TEMPLATE_FORMAT_V1, handler);
            handlers.put(DiagramTemplateConstants.DRAWING_TEMPLATE_FORMAT_V2, handler);
            return DataContainers.readFile(modelFile, handlers);
        } catch (DataFormatException e) {
            throw new IOException(e);
        } catch (Exception e) {
            if (e instanceof RuntimeException)
                throw (RuntimeException)e;
            else
                throw new IOException(e);
        }
    }

    private abstract static class DiagramTemplateFormatHandler implements FormatHandler<Resource> {
        @Override
        public Binding getBinding() {
            return TransferableGraph1.BINDING;
        }
    }

    private static class TGMonitor implements TGStatusMonitor {
        private final SubMonitor mon;
        private int last = 0;
        public TGMonitor(SubMonitor mon) {
            this.mon = mon;
        }
        @Override
        public void status(int percentage) {
            if (percentage > last) {
                mon.worked(percentage - last);
                last = percentage;
            }
        }
        @Override
        public boolean isCanceled() {
            return mon.isCanceled();
        }
    }

}
