/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.services;

import java.util.List;
import java.util.Set;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.project.IProject;
import org.simantics.project.IProjectService;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintContext.KeyOf;

/**
 * Defines a strategy for naming components in a normal structural model.
 * 
 * <p>
 * The purpose of this interface is to allow different naming conventions for
 * different modelling domains. Almost always it is necessary to make names
 * unique within a certain context. The context may be either the whole model
 * (all structural levels) or perhaps within a single composite component.
 * 
 * <p>
 * To put your own naming strategy into use, please set it into the active
 * {@link IProject} using the {@link #PROJECT_KEY} hint key.
 * 
 * <p>
 * Implementations must be thread-safe. Implementations should not return the
 * same name twice, a least not within a small time-window. Since holding a set
 * of all returned names cannot be held in memory indefinitely, names that
 * have been returned but are not actually in use may be reused.
 * 
 * @author Tuukka Lehtonen
 */
public interface ComponentNamingStrategy extends IProjectService {

    /**
     * A key used for storing a project-local {@link ComponentNamingStrategy}
     * within an {@link IProject}.
     */
    Key PROJECT_KEY = new KeyOf(ComponentNamingStrategy.class);

    /**
     * Finds a unique name for a new instance of the specified component type
     * within the context of the specified configuration element.
     * 
     * @param graph
     * @param configurationRoot
     * @param container
     * @param componentType
     * @return a unique component name within the specified context and the
     *         specified component type
     * @throws NamingException if a fresh name cannot be resolved, possibly out
     *         of names
     * @throws DatabaseException in case of any database errors
     */
    String findFreshInstanceName(
            ReadGraph graph,
            Resource configurationRoot,
            Resource container,
            Resource componentType)
    throws NamingException, DatabaseException;

    /**
     * Validates the given proposition for the given component
     * 
     * @param graph
     * @param configurationRoot
     * @param component
     * @param proposition
     * @return a unique component name within the specified context based on the
     *         specified name proposition and possibly the component type
     * @throws NamingException if the specified name is invalid and cannot be
     *         made valid
     * @throws DatabaseException in case of any database errors
     */
    String validateInstanceName(
            ReadGraph graph,
            Resource configurationRoot,
            Resource component,
            String proposition,
            boolean acceptProposition)
    throws NamingException, DatabaseException;
    
    /**
     * Validates the specified name proposition. Finds a unique name for a new
     * instance of the specified component type within the context of the
     * specified configuration element.
     * 
     * @param graph
     * @param configurationRoot
     * @param container
     * @param componentType
     * @param proposition
     * @return a unique component name within the specified context based on the
     *         specified name proposition and possibly the component type
     * @throws NamingException if the specified name is invalid and cannot be
     *         made valid
     * @throws DatabaseException in case of any database errors
     */
    String validateInstanceName(
            ReadGraph graph,
            Resource configurationRoot,
            Resource container,
            Resource componentType,
            String proposition)
    throws NamingException, DatabaseException;

    /**
     * Validates the specified name proposition. Finds a unique name for a new
     * instance of the specified component type within the context of the
     * specified configuration element.
     * 
     * @param graph
     * @param configurationRoot
     * @param container
     * @param componentType
     * @param proposition name proposition to validate
     * @param acceptProposition <code>true</code> if proposition as such is
     *        considered an accepted value, <code>false</code> if not
     * @return a unique component name within the specified context based on the
     *         specified name proposition and possibly the component type
     * @throws NamingException if the specified name is invalid and cannot be
     *         made valid
     * @throws DatabaseException in case of any database errors
     */
    String validateInstanceName(
            ReadGraph graph,
            Resource configurationRoot,
            Resource container,
            Resource componentType,
            String proposition,
            boolean acceptProposition)
    throws NamingException, DatabaseException;

    /**
     * Validates the specified list of name propositions. Finds a unique name
     * for each listed proposition so that the returned names don't collide with
     * existing names in the model configuration and they don't collide with
     * each other.
     * <p>
     * This method shall not reserve the returned names so that later
     * invocations to any methods in this interface would not return the same
     * names. It is the callers responsibility to handle that this method is
     * either invoked within a write transaction to ensure that no other party
     * validates names simultaneously or otherwise make sure that there are no
     * problems through UI design.
     * 
     * @param graph
     * @param configurationRoot
     * @param propositions
     *            list of name propositions to validate
     * @param acceptProposition
     *            <code>true</code> if proposition as such is considered an
     *            accepted value, <code>false</code> if not
     * @param externallyReserved
     *            a collection of names that are to be considered externally
     *            reserved that the validated names shall not collide with. May
     *            be <code>null</code> to specify no externally reserved names.
     * @return unique names to match each provided proposition in the same
     *         order. The returned names shall not collide with existing names
     *         in the configuration nor each other.
     * @throws NamingException
     *             if the specified name is invalid and cannot be made valid
     * @throws DatabaseException
     *             in case of any database errors
     */
    List<String> validateInstanceNames(
            ReadGraph graph,
            Resource configurationRoot,
            List<String> propositions,
            boolean acceptProposition,
            Set<String> externallyReserved)
    throws NamingException, DatabaseException;

}
