package org.simantics.modeling.mapping;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Set;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.impl.DefaultPasteHandler;
import org.simantics.db.layer0.util.ClipboardUtils;
import org.simantics.db.layer0.util.PasteEventHandler;
import org.simantics.db.layer0.util.SimanticsClipboard.Representation;
import org.simantics.db.layer0.util.SimanticsKeys;
import org.simantics.graph.representation.TransferableGraph1;

/**
 * Paste handler for Component and it's Element.
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public abstract class ComponentAndElementPasteHandler extends DefaultPasteHandler{

	protected boolean includeComponent = true;
	protected boolean includeElement = true;
	
	/**
	 * Creates Paste Handler for target composite (Composite where the data is copied).
	 * @param composite
	 */
	public ComponentAndElementPasteHandler(Resource composite) {
		super(composite);
	}
	
	/**
	 * Creates Paste Handler for target composite (Composite where the data is copied).
	 * @param composite
	 * @param includeComponent true if component is copied. (component must exist in the source data).
	 * @param includeElement true if element is copied. If the element is not included in the source data and this flag is true, the element is created. 
	 */
	public ComponentAndElementPasteHandler(Resource composite, boolean includeComponent, boolean includeElement) {
		super(composite);
		if (!includeComponent && !includeElement)
			throw new IllegalArgumentException();
		this.includeElement = includeElement;
		this.includeComponent = includeComponent;
		
	}

	/**
	 * Returns configuration composite
	 * 
	 * TODO: Current Modeling, Structural, or Diagram Ontologies do not have specific relation/type for configuration, so it is impossible to create generic implementation.
	 * 
	 * @param graph
	 * @return
	 * @throws DatabaseException
	 */
	abstract protected  Resource getConfiguration(ReadGraph graph) throws DatabaseException;
	
	public void onPaste(WriteGraph graph, ComponentAndElementPasteImportAdvisor advisor, Set<Representation> representations) throws DatabaseException {
		
	}
	
	/**
	 * Called if the PasteImportAdviosr had to create the element.
	 * This happens if an element was required, but the copied data did not include the element.
	 * 
	 * Override this method if your element requires custom data. 
	 *    
	 * @param graph
	 * @param element
	 * @throws DatabaseException
	 */
	protected void createdElementInAdvisor(WriteGraph graph, Resource element) throws DatabaseException {
		
	}
	
	@Override
	public Collection<Resource> pasteObject(WriteGraph graph, Set<Representation> object, PasteEventHandler handler) throws DatabaseException {
		Collection<Resource> result = new ArrayList<Resource>();
		
        TransferableGraph1 tg = ClipboardUtils.accept(graph, object, SimanticsKeys.KEY_TRANSFERABLE_GRAPH);
        if (tg != null) {

        	doPaste(graph, object, tg, result);
        }
        return result;
    }
	
	public void doPaste(WriteGraph graph, Set<Representation> object, TransferableGraph1 tg, Collection<Resource> result) throws DatabaseException{
		Resource resource = (Resource)getAdapter(Resource.class);
		Resource configuration = getConfiguration(graph);
        ComponentAndElementPasteImportAdvisor advisor = new ComponentAndElementPasteImportAdvisor(graph, configuration, includeComponent, includeElement) {
        	@Override
        	public void createElement(WriteGraph graph, Resource element)
        			throws DatabaseException {
        		createdElementInAdvisor(graph, element);
        	}
        }; 
        execute(graph, tg, resource, advisor);
        advisor.attach(graph);
        onPaste(graph, advisor, object);
        if (advisor.getComponent() != null)
        	result.add(advisor.getComponent());
        else if (advisor.getElement() != null)
        	result.add(advisor.getElement());
	}
	
	
	
	
}
