package org.simantics.modeling.adapters;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.simantics.browsing.ui.model.children.ChildRule;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.modeling.ModelingResources;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.function.Function1;

public class SCLChildRule implements ChildRule {

    private Function1<Object,List<Resource>> getChildrenF;
    private Function1<Object,Boolean> hasChildrenF;

    public SCLChildRule(ReadGraph graph, Resource rule) throws DatabaseException {
        ModelingResources MOD = ModelingResources.getInstance(graph);
        Variable ruleVariable = Variables.getVariable(graph, rule);
        this.getChildrenF = ruleVariable.getPropertyValue(graph, MOD.SCLChildRule_getChildren);
        this.hasChildrenF = ruleVariable.getPossiblePropertyValue(graph, MOD.SCLChildRule_hasChildren);
    }

    @Override
    public boolean isCompatible(Class<?> contentType) {
        return contentType.equals(Resource.class) || contentType.equals(Variable.class);
    }

    private Resource inputToResource(ReadGraph graph, Object parent) throws DatabaseException {
        if (parent instanceof Variable) {
            return ((Variable) parent).getRepresents(graph);
        } else {
            return (Resource) parent;
        }
    }

    @Override
    public Collection<Resource> getChildren(ReadGraph graph, Object parent) throws DatabaseException {
        if (getChildrenF == null)
            return Collections.emptyList();
        return apply(graph, getChildrenF, inputToResource(graph, parent));
    }

    @Override
    public boolean hasChildren(ReadGraph graph, Object parent) throws DatabaseException {
        Resource r = inputToResource(graph, parent);
        if (hasChildrenF != null)
            return apply(graph, hasChildrenF, r);
        return !apply(graph, getChildrenF, r).isEmpty();
    }

    private <P,T> T apply(ReadGraph graph, Function1<P, T> f, P parameter) throws DatabaseException {
        SCLContext sclContext = SCLContext.getCurrent();
        Object oldGraph = sclContext.get("graph");
        try {
            sclContext.put("graph", graph);
            return f.apply(parameter);
        } catch (Throwable t) {
            throw new DatabaseException(t);
        } finally {
            sclContext.put("graph", oldGraph);
        }
    }

    @Override
    public Collection<?> getParents(ReadGraph graph, Object child) throws DatabaseException {
        return Collections.emptyList();
    }

}
