package org.simantics.issues.ui.property;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.ViewerCell;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IWorkbenchSite;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupport;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.request.Read;
import org.simantics.issues.Severity;
import org.simantics.issues.common.IssueResourceContexts;
import org.simantics.issues.common.ListModelIssuesBySeverity;
import org.simantics.issues.common.SimpleIssue;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.selectionview.PropertyTabContributorImpl;

/**
 * A very simple property tab for listing issues of a selected object.
 * 
 * @author luukkainen
 *
 */
public class IssuesTab extends PropertyTabContributorImpl{
	
	
	private TableViewer viewer;
	private Resource model;
	private Resource element;
	
	private List<SimpleIssue> result;

	public IssuesTab(ReadGraph graph, Resource model, Resource element) throws DatabaseException {
		this.model = model;
		this.element = element;

		Layer0 L0 = Layer0.getInstance(graph);
		List<Resource> contexts = getContexts(graph, element);
		Map<Severity, List<Resource>> issuesBySeverity = graph.syncRequest(
				new ListModelIssuesBySeverity(model, true, true, Severity.NOTE),
				TransientCacheListener.<Map<Severity, List<Resource>>>instance());

		result = new ArrayList<>();
		for (Severity severity : Severity.values()) {
			List<Resource> issues = issuesBySeverity.get(severity);
			if (issues != null) {
				for (Resource issue : issues) {
					Set<Resource> issueContexts = graph.syncRequest(new IssueResourceContexts(issue));
					if (!Collections.disjoint(issueContexts, contexts)) {
						SimpleIssue si = new SimpleIssue((String) graph.getRelatedValue(issue, L0.HasLabel, Bindings.STRING),
								severity, issue);
						result.add(si);
						System.out.println(si.label + " " + si.severity);
					}
				}
			}
		} 
	}
	
	
	/**
	 * @see org.simantics.modeling.ui.diagram.style.IssueDecorationStyle
	 * 
	 * @param graph
	 * @param element
	 * @return
	 * @throws DatabaseException
	 */
	protected List<Resource> getContexts(ReadGraph graph, Resource element) throws DatabaseException {

        ModelingResources MOD = ModelingResources.getInstance(graph);
        List<Resource> result = new ArrayList<Resource>(3);
        result.add(element);
        Resource config = graph.getPossibleObject(element, MOD.ElementToComponent);
        if (config != null && result.indexOf(config) == -1) result.add(config);
        config = graph.getPossibleObject(element, MOD.DiagramConnectionToConnection);
        if (config != null && result.indexOf(config) == -1) result.add(config);
        // For diagram reference element support
        config = graph.getPossibleObject(element, MOD.HasParentComponent);
        if (config != null && result.indexOf(config) == -1) result.add(config);
        return result;

    }
	
	
	@Override
	public void createControls(Composite body, IWorkbenchSite site, ISessionContext context, WidgetSupport support) {
		GridLayoutFactory.fillDefaults().numColumns(2).spacing(0, 0).applyTo(body);
	
		viewer = new TableViewer(body);
		
		TableViewerColumn issueColumn = new TableViewerColumn(viewer, SWT.LEFT);
		TableViewerColumn severityColumn = new TableViewerColumn(viewer, SWT.LEFT);
		
		issueColumn.setLabelProvider(new CellLabelProvider() {
			
			@Override
			public void update(ViewerCell cell) {
				SimpleIssue issue = (SimpleIssue)cell.getElement();
				cell.setText(issue.label);
				
			}
		});
		severityColumn.setLabelProvider(new CellLabelProvider() {
			
			@Override
			public void update(ViewerCell cell) {
				SimpleIssue issue = (SimpleIssue)cell.getElement();
				cell.setText(issue.severity.name());
			}
		});
		
		issueColumn.getColumn().setWidth(300);
		issueColumn.getColumn().setText("Issue");
		
		severityColumn.getColumn().setWidth(100);
		severityColumn.getColumn().setText("Severity");
		
		viewer.getTable().setHeaderVisible(true);
		viewer.getTable().setLinesVisible(true);
		
		viewer.setContentProvider(new IStructuredContentProvider() {
			
			@Override
			public Object[] getElements(Object inputElement) {
				List<SimpleIssue> list = (List<SimpleIssue>)inputElement;
				return list.toArray();
			}
		});
			
		viewer.setInput(result);
		GridDataFactory.fillDefaults().grab(true, true).applyTo(viewer.getControl());
		
	}
	
	@Override
	public Read<String> getPartNameReadRequest(ISelection forSelection) {
		return new ObjectTitleRead(element);
	}
	
	public static class ObjectTitleRead extends ResourceRead<String> {
		public ObjectTitleRead(Resource resource) {
			super(resource);
		}

		@Override
		public String perform(ReadGraph graph) throws DatabaseException {
			return NameUtils.getSafeName(graph, resource);
		}
	}

}
