/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - GitLab #1134
 *******************************************************************************/
package org.simantics.issues.common;

import java.util.Collection;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.common.utils.ListUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.function.DbBiFunction;
import org.simantics.issues.Severity;
import org.simantics.issues.ontology.IssueResource;

/**
 * @author Tuukka Lehtonen
 */
public class MaxIssueSeveritySingle extends ResourceRead<Severity>{

    public MaxIssueSeveritySingle(Resource resource) {
        super(resource);
    }

    @Override
    public Severity perform(ReadGraph graph) throws DatabaseException {
        //System.out.println("severityFor: " + NameUtils.getSafeName(graph, resource));
        IssueResource ISSUE = IssueResource.getInstance(graph);
        Severity maxSeverity = null;

        DbBiFunction<Resource, Severity, Severity> newSeverity = (Resource issue, Severity previousMaxSeverity) -> {
            boolean resolved = graph.hasStatement(issue, ISSUE.Resolved);
            if (resolved)
                return previousMaxSeverity;

            Resource issueSource = graph.getPossibleObject(issue, ISSUE.IssueSource_Manages_Inverse);
            if (issueSource != null) {
                Boolean active = graph.getPossibleRelatedValue(issueSource, ISSUE.IssueSource_active, Bindings.BOOLEAN);
                if (Boolean.FALSE.equals(active)) {
                    return previousMaxSeverity;
                }
            } else {
                // Not managed by an issue source => user issue
            }

            Resource severity = graph.getPossibleObject(issue, ISSUE.Issue_HasSeverity);
            return severity != null
                    ? Severity.moreSevere(previousMaxSeverity, toSeverity(ISSUE, severity))
                    : previousMaxSeverity;
        };

        Collection<Resource> issues = graph.getObjects(resource, ISSUE.Issue_HasContext_Inverse);
        for (Resource issue : issues) {
            maxSeverity = newSeverity.apply(issue, maxSeverity);
        }

        Collection<Resource> elements = graph.getObjects(resource, ISSUE.Issue_ContextList_Element_Inverse);
        for (Resource element : elements) {
            Resource list = ListUtils.getListElementList(graph, element); 
            Resource issue = graph.getPossibleObject(list, ISSUE.Issue_HasContexts_Inverse);
            if (issue == null)
                continue;
            maxSeverity = newSeverity.apply(issue, maxSeverity);
        }

        //System.out.println("severityFor: " + NameUtils.getSafeName(graph, resource) + " : " + maxSeverity);
        return maxSeverity;
    }

    private static Severity toSeverity(IssueResource ISSUE, Resource severity) {
        if (ISSUE.Severity_Fatal.equals(severity))
            return Severity.FATAL;
        if (ISSUE.Severity_Error.equals(severity))
            return Severity.ERROR;
        if (ISSUE.Severity_Warning.equals(severity))
            return Severity.WARNING;
        if (ISSUE.Severity_Info.equals(severity))
            return Severity.INFO;
        if (ISSUE.Severity_Note.equals(severity))
            return Severity.NOTE;
        return null;
    }

}
