package org.simantics.issues.common;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.simantics.db.Issue;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.QueryMemoryWatcher;
import org.simantics.db.common.primitiverequest.IsInheritedFrom;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.common.utils.Functions;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.ModelTransferableGraphSourceRequest.DomainOnlyProcessor;
import org.simantics.db.service.QueryControl;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.layer0.Layer0;

public class DependencyTrackerBatchIssueSource implements BatchIssueSource {

	final Resource source;
	
	public DependencyTrackerBatchIssueSource(Resource source) {
		this.source = source;
	}
	
	@Override
	public Map<Resource, Set<Issue>> run(IProgressMonitor monitor, ReadGraph graph, BatchIssueValidationContext context) throws DatabaseException {
		
        SubMonitor progress = SubMonitor.convert(monitor);

		Layer0 L0 = Layer0.getInstance(graph);
		IssueResource IR = IssueResource.getInstance(graph);
		
		Resource constraint = graph.getSingleObject(source, IR.Sources_DependencyTracker_HasConstraint);
		Resource type = graph.getSingleObject(constraint, L0.HasConstraint_Inverse);
		Resource function = graph.getSingleObject(constraint, L0.Constraint_Validator);
		
		String sourceName = NameUtils.getSafeName(graph, source);

        QueryControl qc = graph.getService(QueryControl.class);
        qc.flush(graph);

        progress.subTask(sourceName);
        progress.worked(1);
        
		HashMap<Resource, Set<Issue>> result = new HashMap<Resource, Set<Issue>>();
		
		// Allow this process to make 50k queries
		QueryMemoryWatcher memory = new QueryMemoryWatcher(graph, 50000, 0.5, 300);

		int number = 0;
		
		DomainOnlyProcessor domain = context.domain;

		for(int i=0;i<domain.internals.size();i++) {

			Resource r = domain.internals.get(i);
			Resource singleType = domain.internalTypes.get(i);

			if (number % 1000 == 0) {
				if(progress.isCanceled()) return Collections.emptyMap();
				memory.maintain();
				progress.subTask(sourceName + " " + (100*number)/domain.internals.size() + "% done.");
			}
			++number;


			if(singleType != null) {
				if(!graph.syncRequest(new IsInheritedFrom(singleType, type), TransientCacheAsyncListener.<Boolean>instance())) continue;
			} else {
				if(!graph.isInstanceOf(r, type)) continue;
			}

			@SuppressWarnings("unchecked")
			List<Issue> is = (List<Issue>)Functions.exec(graph, function, graph, r);
			if(is.isEmpty())
				result.put(r, Collections.<Issue>emptySet());
			else
				result.put(r, new HashSet<Issue>(is));

		}

		return result;

	}

	@Override
	public Resource getResource() {
		return source;
	}

}
