package org.simantics.graph.compiler.internal.resourceFiles;

import gnu.trove.set.hash.THashSet;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.adapter.AdaptException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.graph.IResourceFile;
import org.simantics.graph.query.Path;
import org.simantics.graph.query.PathChild;
import org.simantics.graph.query.Paths;
import org.simantics.graph.store.GraphStore;

public class ResourceFileGenerator {
	public static Collection<IResourceFile> generate(Paths paths, GraphStore store) {
		int HasResourceClass = store.identities.pathToId(paths.HasResourceClass);
		if(HasResourceClass < 0)
			return Collections.emptyList();
		
		Collection<IResourceFile> result = new ArrayList<IResourceFile>();
		
		int resourceCount = store.identities.getResourceCount();
		for(int ontology=0;ontology<resourceCount;++ontology) {
			for(int v : store.statements.getObjects(ontology, HasResourceClass).toArray()) {
				Variant value = store.values.getByteValue(v);
				if(value == null)
					continue;
				try {
					String className = (String)value.getValue(Bindings.STRING);
					result.add(generate(paths, store, ontology, className));
				} catch (AdaptException e) {
                }
			}
		}
		
		return result;
	}

	private static ResourceFile generate(Paths paths,
	        GraphStore store,
			int ontology,
			String fullClassName) {
		int p = fullClassName.lastIndexOf('.');
		String packageName;
		String className;
		if(p>0) {
			packageName = fullClassName.substring(0, p);
			className = fullClassName.substring(p+1);
		}
		else {
			packageName = "";
			className = fullClassName;
		}
		
		List<ResourceRef> resources = new ArrayList<ResourceRef>(); 
		findResources(store,
				store.identities.pathToId(paths.Deprecated),
				store.identities.idToPath(ontology),
				ontology, 
				resources
			);
		
		ResourceFile file = new ResourceFile(packageName, className, resources);
		file.sort();
		return file;
	}

	private static void findResources(GraphStore store,
			int deprecatedId,
			Path root,
			int parent,
			List<ResourceRef> resources) {
		for(int child : store.identities.getChildren(parent)) {
			ResourceRef ref = createResource(root, store.identities.idToPath(child));
			if(!store.statements.getObjects(child, deprecatedId).isEmpty())
				ref.deprecated = true;
			resources.add(ref);
			findResources(store, deprecatedId, root, child, resources);
		}
	}
	
	static THashSet<String> KEYWORDS = new THashSet<String>();
	
	static {
		for(String s : new String[] {
				"abstract", "continue", "for", "new", "switch",
				"assert", "default", "goto", "package", "synchronized",
				"boolean", "do", "if", "private", "this",
				"break", "double", "implements", "protected", "throw",
				"byte", "else", "import", "public", "throws",
				"case", "enum", "instanceof", "return", "transient",
				"catch", "extends", "int", "short", "try",
				"char", "final", "interface", "static", "void",
				"class", "finally", "long", "strictfp", "volatile",
				"const", "float", "native", "super", "while",
				"true", "false", "null"
		})
			KEYWORDS.add(s);
	}

	private static ResourceRef createResource(Path root, Path path) {
		StringBuilder b = new StringBuilder();
		javaName(b, root, path);
		String javaName = b.toString();
		if(KEYWORDS.contains(javaName))
			javaName = javaName + "_";
		else if(!Character.isJavaIdentifierStart(javaName.charAt(0)))
			javaName = "_" + javaName;
		return new ResourceRef(javaName, path.toString());
	}
	
	private static void javaName(StringBuilder b, Path root, Path path) {
		if(!root.equals(path)) {
			PathChild pc = (PathChild)path;
			javaName(b, root, pc.parent);
			if(b.length() > 0)
				b.append('_');
			for(int i=0;i<pc.name.length();++i) {
				char c = pc.name.charAt(i);
				if(Character.isJavaIdentifierPart(c))
					b.append(c);
				else if(c==' ')
					b.append('_');
				else
					b.append('$');
			}
		}		
	}
}
