/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.layers;

import java.util.Collections;
import java.util.Comparator;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.CopyOnWriteArrayList;

import org.simantics.utils.strings.AlphanumComparator;

/**
 * @author Antti Villberg
 */
public class SimpleLayers implements ILayersEditor {

    private static final Comparator<ILayer> LAYER_COMPARATOR = new Comparator<ILayer>() {
        @Override
        public int compare(ILayer o1, ILayer o2) {
            return AlphanumComparator.COMPARATOR.compare(o1.getName(), o2.getName());
        }
    };

    private Set<ILayer>                                 all                      = new TreeSet<ILayer>(LAYER_COMPARATOR);
    private volatile Set<ILayer>                        allSnapshot;

    private Set<ILayer>                                 visible                  = new TreeSet<ILayer>(LAYER_COMPARATOR);
    private volatile Set<ILayer>                        visibleSnapshot;

    private CopyOnWriteArrayList<ILayersEditorListener> layerEditorListeners     = new CopyOnWriteArrayList<ILayersEditorListener>();
    private CopyOnWriteArrayList<ILayersListener>       layersListeners          = new CopyOnWriteArrayList<ILayersListener>();

    private boolean                                     ignoreFocusSettings      = false;

    private boolean                                     ignoreVisibilitySettings = false;

    private boolean                                     focusImages              = false;

    public SimpleLayers() {
    }

    public SimpleLayers(String... content) {
        for (String s : content) {
            ILayer layer = new SimpleLayer(s);
            all.add(layer);
            visible.add(layer);
        }
    }
    
    @Override
    public void update(Set<ILayer> allLayers, Set<ILayer> visibleLayers) {
        synchronized (this) {
            all.clear();
            all.addAll(allLayers);
            visible.clear();
            visible.addAll(visibleLayers);
            allSnapshot = null;
            visibleSnapshot = null;
        }
        for (ILayersListener listener : layersListeners) {
            listener.changed();
        }
    }

    @Override
    public Set<ILayer> getLayers() {
        // Double-checked locking idiom: http://en.wikipedia.org/wiki/Double-checked_locking
        // Works with acquire/release semantics for volatile
        // Broken under Java 1.4 and earlier semantics for volatile
        if (allSnapshot == null) {
            synchronized (this) {
                if (allSnapshot == null) {
                    TreeSet<ILayer> ss = new TreeSet<ILayer>(LAYER_COMPARATOR);
                    ss.addAll(all);
                    allSnapshot = Collections.unmodifiableSet(ss);
                }
            }
        }
        return allSnapshot;
    }

    @Override
    public Set<ILayer> getVisibleLayers() {
        // Double-checked locking idiom: http://en.wikipedia.org/wiki/Double-checked_locking
        // Works with acquire/release semantics for volatile
        // Broken under Java 1.4 and earlier semantics for volatile
        if (visibleSnapshot == null) {
            synchronized (this) {
                if (visibleSnapshot == null) {
                    TreeSet<ILayer> ss = new TreeSet<ILayer>(LAYER_COMPARATOR);
                    ss.addAll(visible);
                    visibleSnapshot = Collections.unmodifiableSet(ss);
                }
            }
        }
        return visibleSnapshot;
    }

    @Override
    public boolean isActive(ILayer layer) {
        synchronized (this) {
            return visible.contains(layer);
        }
    }

    @Override
    public void deactivate(ILayer layer) {
        boolean deactivated = false;
        synchronized (this) {
            deactivated = visible.remove(layer);
        }
        if (deactivated) {
            synchronized (this) {
                visibleSnapshot = null;
            }
            for (ILayersListener listener : layersListeners) {
                listener.changed();
            }
            for (ILayersEditorListener listener : layerEditorListeners) {
                listener.layerDeactivated(layer);
            }
        }
    }

    @Override
    public void activate(ILayer layer) {
        boolean activated = false;
        synchronized (this) {
            activated = visible.add(layer);
        }
        if (activated) {
            synchronized (this) {
                visibleSnapshot = null;
            }
            for (ILayersListener listener : layersListeners) {
                listener.changed();
            }
            for (ILayersEditorListener listener : layerEditorListeners) {
                listener.layerActivated(layer);
            }
        }
    }

    @Override
    public void addLayer(ILayer layer) {
        boolean added = false;
        synchronized (this) {
            added = all.add(layer);
        }
        if (added) {
            synchronized (this) {
                allSnapshot = null;
            }
            for (ILayersListener listener : layersListeners) {
                listener.changed();
            }
            for (ILayersEditorListener listener : layerEditorListeners) {
                listener.layerAdded(layer);
            }
        }
    }

    @Override
    public void removeLayer(ILayer layer) {
        boolean removed = false;
        synchronized (this) {
            removed = all.remove(layer);
            visible.remove(layer);
        }
        if (removed) {
            synchronized (this) {
                allSnapshot = null;
                visibleSnapshot = null;
            }
            for (ILayersListener listener : layersListeners) {
                listener.changed();
            }
            for (ILayersEditorListener listener : layerEditorListeners) {
                listener.layerRemoved(layer);
            }
        }
    }

    @Override
    public void addLayerEditorListener(ILayersEditorListener listener) {
        layerEditorListeners.add(listener);
    }

    @Override
    public void removeLayerEditorListener(ILayersEditorListener listener) {
        layerEditorListeners.remove(listener);
    }

    @Override
    public void addLayersListener(ILayersListener listener) {
        layersListeners.add(listener);
    }

    @Override
    public void removeLayersListener(ILayersListener listener) {
        layersListeners.remove(listener);
    }

    @Override
    public boolean getIgnoreFocusSettings() {
        return ignoreFocusSettings;
    }

    @Override
    public void setIgnoreFocusSettings(boolean value) {
        boolean changed = ignoreFocusSettings != value;
        ignoreFocusSettings = value;
        if (changed) {
            for (ILayersListener listener : layersListeners) {
                listener.changed();
            }
            for (ILayersEditorListener listener : layerEditorListeners) {
                listener.ignoreFocusChanged(value);
            }
        }
    }

    @Override
    public boolean getIgnoreVisibilitySettings() {
        return ignoreVisibilitySettings;
    }

    @Override
    public void setIgnoreVisibilitySettings(boolean value) {
        boolean changed = ignoreVisibilitySettings != value;
        ignoreVisibilitySettings = value;
        if (changed) {
            for (ILayersListener listener : layersListeners) {
                listener.changed();
            }
            for (ILayersEditorListener listener : layerEditorListeners) {
                listener.ignoreVisibilityChanged(value);
            }
        }
    }

    @Override
    public boolean getFocusImages() {
        return focusImages;
    }

    @Override
    public void setFocusImages(boolean value) {
        boolean changed = focusImages != value;
        focusImages = value;
        if (changed) {
            for (ILayersListener listener : layersListeners) {
                listener.changed();
            }
            for (ILayersEditorListener listener : layerEditorListeners) {
                listener.focusImagesChanged(value);
            }
        }
    }

}
