/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.elementclass;

import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;

import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.SceneGraphNodeKey;
import org.simantics.g2d.element.handler.InternalSize;
import org.simantics.g2d.element.handler.Outline;
import org.simantics.g2d.element.handler.Resize;
import org.simantics.g2d.element.handler.SceneGraph;
import org.simantics.g2d.element.handler.impl.DefaultTransform;
import org.simantics.g2d.image.Image;
import org.simantics.scenegraph.Node;
import org.simantics.scenegraph.g2d.G2DParentNode;
import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.utils.datastructures.hints.IHintContext.Key;

/**
 * See {@link ElementHints}.KEY_IMAGE
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 * @author Tuukka Lehtonen
 */
public class ImageClass {

    public static final ElementClass INSTANCE =
        ElementClass.compile(
                ImageElementHandler.INSTANCE,
                DefaultTransform.INSTANCE
        );

    public static class ImageElementHandler implements SceneGraph, InternalSize, Resize, Outline {

        private static final long serialVersionUID = 7980198654721917029L;

        public static final ImageElementHandler INSTANCE = new ImageElementHandler();

        public static final Key  KEY_SG_NODE             = new SceneGraphNodeKey(Node.class, "IMAGE_SG_NODE");

        protected Image getImage(IElement e) {
            Image i = e.getHint(ElementHints.KEY_IMAGE);
            if (i == null)
                throw new IllegalStateException("element " + e + " has no ElementHints.KEY_IMAGE hint");
            return i;
        }

        protected Key getNodeKey() {
            return KEY_SG_NODE;
        }

        @Override
        public void init(final IElement e, final G2DParentNode parent) {
            Image i = getImage(e);
            Node node = i.init(parent);
            if (node != null)
                e.setHint(getNodeKey(), node);
            if(node instanceof IG2DNode) {
            	IG2DNode n = (IG2DNode)node;
            	AffineTransform at = ElementUtils.getTransform(e);
            	if(at != null) {
            		n.setTransform(at); // FIXME: not tested..
            	}
            }
        }

        public void cleanup(final IElement e) {
            Node node = e.removeHint(getNodeKey());
            if (node != null)
                node.remove();
        }

        @Override
        public Rectangle2D getBounds(IElement e, Rectangle2D size) {
            Image i = getImage(e);
            Rectangle2D r = i.getBounds();
            if (size == null)
                size = new Rectangle2D.Double();
            size.setFrame(r);
            return size;
        }

        @Override
        public Double getFixedAspectRatio(IElement e) {
            Image i = getImage(e);
            Rectangle2D r = i.getBounds();
            return r.getWidth() / r.getHeight();
        }

        @Override
        public Rectangle2D getMaximumSize(IElement e) {
            Image i = getImage(e);
            return i.getBounds();
        }

        @Override
        public Rectangle2D getMinimumSize(IElement e) {
            Image i = getImage(e);
            return i.getBounds();
        }

        @Override
        public void resize(IElement e, Rectangle2D newSize) {
        }

        @Override
        public Shape getElementShape(IElement e) {
            Image i = getImage(e);
            return i.getOutline();
        }
    }


    public static class StaticImageElementHandler extends ImageElementHandler {
        private static final long serialVersionUID = 7691329107125520049L;

        public static final Key   KEY_SG_NODE      = new SceneGraphNodeKey(Node.class, "STATIC_IMAGE_SG_NODE");

        private final Image i;

        public StaticImageElementHandler(Image i) {
            this.i = i;
        }

        @Override
        protected Image getImage(IElement e) {
            return i;
        }

        @Override
        protected Key getNodeKey() {
            return KEY_SG_NODE;
        }
    }

}