/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.element;

import java.util.HashMap;
import java.util.Map;

/**
 * An IElementClassProvider provides an abstract way of requesting an
 * ElementClass for a specifiable key. The key is purposefully left abstract.
 * 
 * @author Tuukka Lehtonen
 * 
 * @see IElementClassProvider
 * @see ElementClasses
 */
public class ElementClassProviders {

    /**
     * Create an element class provider that provides only a single static
     * ElementClass regardless of the specified key argument.
     * 
     * @param clazz the static element class to provide
     * @return a provider for the specified element class
     */
    public static IElementClassProvider staticProvider(final ElementClass clazz) {
        return new IElementClassProvider() {
            @Override
            public ElementClass get(Object key) {
                return clazz;
            }
        };
    }

    /**
     * Create an element class provider that based on the specified map. The
     * provider will directly access the map with the received keys. The
     * argument map will be copied.
     * 
     * @param map the map to use for element class provision
     * @return <code>null</code> if there is no provider for the specified key
     */
    public static IElementClassProvider mappedProvider(Map<Object, ElementClass> map) {
        // Copy the map as a safety measure
        final Map<Object, ElementClass> copy = new HashMap<Object, ElementClass>(map);
        return new IElementClassProvider() {
            @Override
            public ElementClass get(Object key) {
                return copy.get(key);
            }
        };
    }

    /**
     * Does the same as {@link #mappedProvider(Map)}, the map is simply provided
     * differently. The specified array must contain
     * <code>[key, ElementClass, key, ElementClass, ...]</code>.
     * 
     * @param map the map to use for element class provision
     * @return <code>null</code> if there is no provider for the specified key
     */
    public static IElementClassProvider mappedProvider(Object... keyClassPairs) {
        if (keyClassPairs.length % 2 != 0)
            throw new IllegalArgumentException();
        Map<Object, ElementClass> map = new HashMap<Object, ElementClass>();
        int n = keyClassPairs.length / 2;
        for (int i = 0; i < n; ++i) {
            Object key = keyClassPairs[i * 2];
            Object elementClass = keyClassPairs[i*2+1];
            if (!(elementClass instanceof ElementClass))
                throw new IllegalArgumentException("not an ElementClass instance: "  + elementClass);
            map.put(key, (ElementClass) elementClass);
        }
        return mappedProvider(map);
    }

}
