/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.event.view.contribution;

import java.math.RoundingMode;
import java.text.NumberFormat;
import java.util.Locale;
import java.util.Map;

import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.simantics.browsing.ui.model.labels.LabelRule;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.event.ontology.EventResource;
import org.simantics.event.view.Constants;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.ArrayMap;
import org.simantics.utils.format.TimeFormat;

/**
 * @author Tuukka Lehtonen
 */
public enum EventLabelRule implements LabelRule {

    INSTANCE;

    public static EventLabelRule get() {
        return INSTANCE;
    }

    private static final NumberFormat secondFormatter;
    private static final TimeFormat timeFormat = new TimeFormat(Double.MAX_VALUE, 3);

    static {
        secondFormatter = NumberFormat.getNumberInstance(Locale.US);
        secondFormatter.setMinimumFractionDigits(2);
        secondFormatter.setMaximumFractionDigits(3);
        secondFormatter.setRoundingMode(RoundingMode.HALF_EVEN);
    }

    public static String formatSeconds(double time) {
        return secondFormatter.format(time);
    }

    /**
     * Formats a time specified as a double value in seconds in format
     * <code>hh:mm:ss.SSS</code>
     * 
     * @param time
     * @return
     */
    public static String formatHMSS(double time) {
        return timeFormat.format(time);
	}

    @Override
    public boolean isCompatible(Class<?> contentType) {
        return contentType.equals(Resource.class);
    }

    public Map<String, String> getLabel(ReadGraph graph, Object content, boolean allInformation) throws DatabaseException {

        IEclipsePreferences chartPreferenceNode = InstanceScope.INSTANCE.getNode( "org.simantics.charts" );
        boolean timeFormat = chartPreferenceNode.get("chart.timeformat", "Time").equals("Time");  // Time/Decimal

        Resource event = (Resource) content;

        Layer0 L0 = Layer0.getInstance(graph);
        EventResource EVENT = EventResource.getInstance(graph);

        String tag = graph.getPossibleRelatedValue(event, EVENT.Event_tag, Bindings.STRING);
        String message = graph.getPossibleRelatedValue(event, EVENT.Event_message, Bindings.STRING);
        if (message == null) {
            message = graph.getPossibleRelatedValue(event, L0.HasLabel, Bindings.STRING);
            if (message == null)
                message = graph.getPossibleRelatedValue(event, L0.HasName, Bindings.STRING);
            if (message == null)
                message = NameUtils.getSafeLabel(graph, event);
        }

        StringBuilder eventType = new StringBuilder();
        Integer eventTypeNumber = graph.getPossibleRelatedValue(event, EVENT.Event_typeNumber);
        Resource et = graph.getPossibleObject(event, EVENT.Event_type);
        if (et != null) {
            if (eventTypeNumber != null)
                eventType.append(eventTypeNumber).append(": ");
            eventType.append(NameUtils.getSafeLabel(graph, et));
        } else if (eventTypeNumber != null) {
            eventType.append(eventTypeNumber);
        }

        double timeNumeric = getTimestamp(graph, event, 0.0);
        String time = getTimestampString(graph, timeNumeric, timeFormat);

        Integer index = graph.getPossibleRelatedValue(event, EVENT.Event_index, Bindings.INTEGER);
        String indexS = "";
        if (index != null)
            indexS = String.valueOf(index);

        double returnTimeNumeric = 0;
        String returnTime = null;
        Resource returnedBy = graph.getPossibleObject(event, EVENT.ReturnedBy);
        if (returnedBy != null) {
            returnTimeNumeric = getTimestamp(graph, returnedBy, 0.0);
            returnTime = getTimestampString(graph, returnTimeNumeric, timeFormat);
        }

        String sourceName = graph.getPossibleRelatedValue(event, EVENT.Event_sourceName);
        String milestone = null;
        String returned = null;

        if (allInformation) {
            returned = returnedBy != null ? "Yes" : "";
            if (returnTime == null)
                returnTime = "";

            milestone = "";
            if (graph.hasStatement(event, EVENT.Milestone)) {
                milestone = graph.getPossibleRelatedValue(event, EVENT.Event_milestoneLabel);
                if (milestone == null) milestone = "?";
            }
        }

        return ArrayMap.make(Constants.COLUMN_KEYS, new String[] {
                indexS,
                time,
                String.valueOf(timeNumeric),
                milestone,
                eventType.toString(),
                returned,
                tag,
                message,
                returnTime,
                String.valueOf(returnTimeNumeric),
                sourceName
        });
    }

    @Override
    public Map<String, String> getLabel(ReadGraph graph, Object content) throws DatabaseException {
        return getLabel(graph, content, false);
    }

    public static double getTimestamp(ReadGraph graph, Resource event, double defaultValue) throws DatabaseException {
        EventResource EVENT = EventResource.getInstance(graph);
        Double d = graph.getPossibleRelatedValue(event, EVENT.HasTimestamp, Bindings.DOUBLE);
        return d != null ? d : defaultValue;
    }

    public static String getTimestampString(ReadGraph graph, double t, boolean timeFormat) {
        return timeFormat ? formatHMSS(t) : formatSeconds(t);
    }

}
