/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import java.util.List;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.primitiverequest.Adapter;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.diagram.synchronization.ErrorHandler;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.diagram.handler.SubstituteElementClass;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.IElement;
import org.simantics.scl.runtime.tuple.Tuple3;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Antti Villberg
 */
public class NodeRequest2 extends BaseRequest2<Resource, Tuple3> {

    private static final Logger LOGGER = LoggerFactory.getLogger(NodeRequest2.class);

    final IDiagram diagram;

    public NodeRequest2(ICanvasContext canvas, IDiagram diagram, Resource resource) {
        super(canvas, resource);
        this.diagram = diagram;
    }

    @Override
    public void perform(AsyncReadGraph graph, final AsyncProcedure<Tuple3> procedure) {
        // Keep this code from crashing the whole DB client by unexpected
        // throwing of NPE's somewhere in the following code that leads to
        // procedure not getting called properly.
        if (diagram == null) {
            procedure.exception(graph, new NullPointerException("null diagram specified for resource " + data));
            return;
        }

        final ErrorHandler eh = ElementFactoryUtil.getErrorHandler(diagram);

        graph.forHasStatement(data, new AsyncProcedure<Boolean>() {

            @Override
            public void exception(AsyncReadGraph graph, Throwable throwable) {
                eh.error("NodeRequest.forHasStatement failed", throwable);
                procedure.execute(graph, null);
            }

            @Override
            public void execute(AsyncReadGraph graph, Boolean result) {

                if(!result) {
                    procedure.execute(graph, null);
                    return;
                }

                graph.asyncRequest(new Adapter<ElementFactory>(data, ElementFactory.class), new TransientCacheAsyncListener<ElementFactory>() {

                    @Override
                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                        eh.error("NodeRequest.asyncRequest(Adapter<ElementFactory>) failed", throwable);
                        procedure.execute(graph, null);
                    }

                    @Override
                    public void execute(AsyncReadGraph graph, final ElementFactory factory) {

                        graph.asyncRequest(new GetElementClassRequest(factory, data, canvas, diagram), new TransientCacheAsyncListener<ElementClass>() {

                            @Override
                            public void exception(AsyncReadGraph graph, Throwable throwable) {
                                LOGGER.error("Unexpected error in GetElementClassRequest", throwable);
                                procedure.execute(graph, null);
                            }

                            @Override
                            public void execute(AsyncReadGraph graph, ElementClass mutableClazz) {
                                List<SubstituteElementClass> substitutes = diagram.getDiagramClass().getItemsByClass(SubstituteElementClass.class);
                                for (SubstituteElementClass subs : substitutes) {
                                    mutableClazz = subs.substitute(diagram, mutableClazz);
                                }
                                final ElementClass clazz = mutableClazz;
                                graph.asyncRequest(new SpawnRequest(canvas, clazz, data), new TransientCacheAsyncListener<IElement>() {

                                    @Override
                                    public void exception(AsyncReadGraph graph, Throwable throwable) {
                                        LOGGER.error("Unexpected error in SpawnRequest", throwable);
                                        procedure.execute(graph, null);
                                    }

                                    @Override
                                    public void execute(AsyncReadGraph graph, IElement element) {
                                        procedure.execute(graph, new Tuple3(element, clazz, factory));
                                    }

                                });

                            }

                        });

                    }

                });

            }

        });

    }

}
