/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.exception;

import java.util.ArrayList;
import java.util.Collection;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;

/**
*
* A base class for exceptions in org.simantics.db.
* 
* There are three classes of DatabaseException
* <ul>
* <li>ServiceException is a class of serious failures e.g. (connection failure, database corruption) 
* <li>ValidationException is a class of exceptions due to invalid semantic graph 
* <li>AssumptionException is a class of exceptions due to unsatisfied user assumptions (e.g. user assumes a single result or existence of an adapter)
* <li>VariableException is a class of exceptions related to Simantics Variable-interface operations
* </ul>
* 
* The resources related to an exception are available for pretty printing uses. 
* 
* @author Antti Villberg
* @version 0.7
* @see ValidationException
* @see ServiceException
* @see AssumptionException
* @see VariableException
* 
*/
public class DatabaseException extends Exception {

    private static final long serialVersionUID = -6234485044648476711L;

    private ArrayList<Resource> resources;
    private ArrayList<String> names;
    private ArrayList<Integer> indices;
    
    public DatabaseException() {
        super();
    }

    public DatabaseException(String message, Throwable cause) {
        super(message, cause);
    }

    public DatabaseException(String message, Throwable cause, Resource ... rs) {
        super(message, cause);
        resources = new ArrayList<Resource>();
        for(Resource r : rs) resources.add(r); 
    }

    public DatabaseException(Throwable cause, Resource ... rs) {
        super(cause);
        resources = new ArrayList<Resource>();
        for(Resource r : rs) resources.add(r); 
    }

    public DatabaseException(String message, Resource ... rs) {
        super(message);
        resources = new ArrayList<Resource>();
        for(Resource r : rs) resources.add(r); 
    }

    public DatabaseException(String message) {
        super(message);
    }

    public DatabaseException(String message, int ... resources) {
        super(message);
        for(int r : resources) addIndex(r);
    }

    public DatabaseException(int ... resources) {
        super();
        for(int r : resources) addIndex(r);
    }

    public DatabaseException(Throwable cause) {
        super(cause);
    }
    
    public void addResource(Resource resource) {

        assert(resource != null);
        
        if(resources == null) resources = new ArrayList<Resource>();
        
        resources.add(resource);
        
    }
    
    public void addIndex(int index) {

        assert(index != 0);
        
        if(indices == null) indices = new ArrayList<Integer>();
        
        indices.add(index);
        
    }

    public int getIndex(int index) {
        return indices.get(index);
    }

    public Collection<Resource> getResources() {
        return resources;
    }
    
    public void setNames(ArrayList<String> names) {
        this.names  = names;
    }
    
    /**
     * Throws the exception caused by the invocation of this request. The
     * parameter allows checked exceptions to be thrown.
     * 
     * @param <T>
     * @param clazz
     * @throws T
     */
    @SuppressWarnings("unchecked")
    public <T extends Throwable> void throwCause(Class<T> clazz) throws T {
        Throwable t = getCause();
        if (t == null)
            return;
        if (clazz.isInstance(t))
            throw (T) t;
        if (t instanceof RuntimeException)
            throw (RuntimeException) t;
        if (t instanceof Error)
            throw (Error) t;
        throw new RuntimeException("See cause for the real exception.", t);
    }

    public DatabaseException newStack() {
    	return new DatabaseException(this);
    }
    
    @Override
    public String toString() {
        if(names != null) System.out.println(names);
        return super.toString();
    }
    
    public String getShortExplanation() {
    	return getMessage();
    }
    
    public String getExplanation(ReadGraph graph) throws DatabaseException {
    	return getShortExplanation();
    }
    
    public void logPossibleRequest() {
    }
    
}
