/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - GitLab #1117
 *******************************************************************************/
package org.simantics.db.layer0.request;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.Template;
import org.simantics.operation.Layer0X;

/**
 * Applies templates to an instance.
 * @author Hannu Niemistö
 */
public class ApplyTemplatesRequest {
	Map<String, Object> parameters;
	Collection<Resource> contexts;
	
	public ApplyTemplatesRequest(Map<String, Object> parameters,
			Collection<Resource> contexts) {
		this.parameters = parameters;
		this.contexts = contexts;
	}
	
	public ApplyTemplatesRequest(Map<String, Object> parameters) {
		this(parameters, new ArrayList<Resource>(2));
	}
	
	public ApplyTemplatesRequest() {
		this(new HashMap<String, Object>());
	}

	public void perform(WriteGraph g, Resource resource) throws DatabaseException {
	    Layer0X L0X = Layer0X.getInstance(g);

		parameters.put("", resource);
		
		// Use default context if no other contexts have been defined
		if(contexts.isEmpty())
			contexts.add(L0X.HasTemplate);
		
		// Collect templates
		var templateResources = new HashSet<Resource>();
		for(Resource context : contexts)
			templateResources.addAll(g.getObjects(resource, context));
		
		// Apply templates in ascending priority order
		var templates = new ArrayList<Template>(templateResources.size());
		for(Resource template : templateResources)
		    templates.add(g.adapt(template, Template.class));
		
		Collections.sort(templates, (t1, t2) -> Double.compare(t1.priority(), t2.priority()));
		
		for (Template template : templates) {
		    template.apply(g, parameters);
		}
	}
	
	public void addContext(Resource context) {
		contexts.add(context);
	}
	
	public void addParameter(String key, Object value) {
		parameters.put(key, value);
	}
	
	@SuppressWarnings("unchecked")
	public <T> T getParameter(String key) {
		return (T)parameters.get(key);
	}
	
	public Map<String,Object> getParameters() {
		return parameters;
	}
}
