/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.adapter.impl;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.parser.repository.DataTypeSyntaxError;
import org.simantics.databoard.parser.repository.DataValueRepository;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.BooleanType;
import org.simantics.databoard.type.ByteType;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.FloatType;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.type.LongType;
import org.simantics.databoard.type.StringType;
import org.simantics.databoard.units.IUnitConverter;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ValidationException;
import org.simantics.db.layer0.adapter.AbstractStringModifier;
import org.simantics.db.layer0.commandlog.ClaimValueCommand;
import org.simantics.db.layer0.util.PrimitiveValueParser;
import org.simantics.db.layer0.util.PrimitiveValueParser.BooleanArrayValidator;
import org.simantics.db.layer0.util.PrimitiveValueParser.BooleanValidator;
import org.simantics.db.layer0.util.PrimitiveValueParser.ByteArrayValidator;
import org.simantics.db.layer0.util.PrimitiveValueParser.DoubleArrayValidator;
import org.simantics.db.layer0.util.PrimitiveValueParser.DoubleValidator;
import org.simantics.db.layer0.util.PrimitiveValueParser.FloatArrayValidator;
import org.simantics.db.layer0.util.PrimitiveValueParser.FloatValidator;
import org.simantics.db.layer0.util.PrimitiveValueParser.IValidator;
import org.simantics.db.layer0.util.PrimitiveValueParser.IntegerArrayValidator;
import org.simantics.db.layer0.util.PrimitiveValueParser.IntegerValidator;
import org.simantics.db.layer0.util.PrimitiveValueParser.LongArrayValidator;
import org.simantics.db.layer0.util.PrimitiveValueParser.LongValidator;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.layer0.Layer0;
import org.simantics.utils.commandlog.Commands;

public final class StringModifierImpl extends AbstractStringModifier {

    final Resource resource;
    final IUnitConverter converter;
    IValidator     validator;

    public StringModifierImpl(ReadGraph g, Resource resource) throws DatabaseException {
        this(g, resource, null);
    }

    public StringModifierImpl(ReadGraph g, Resource resource, IUnitConverter converter) throws DatabaseException {
    	super(resource);
        this.resource = resource;
        this.validator = null;
        this.converter = converter;

        // Resolve validator if any
        Layer0 l0 = Layer0.getInstance(g);
        validator = PrimitiveValueParser.NON_MODIFIABLE_VALIDATOR;

        Datatype type = g.getPossibleRelatedValue(resource, l0.HasDataType,
                Bindings.getBindingUnchecked(Datatype.class));
        if (type == null)
            return;

        if (type instanceof DoubleType) {
            validator = new DoubleValidator();
        } else if (type instanceof StringType) {
            validator = null;
        } else if (type instanceof ByteType) {
            validator = new ByteArrayValidator();
        } else if (type instanceof BooleanType) {
            validator = new BooleanValidator();
        } else if (type instanceof IntegerType) {
            validator = new IntegerValidator();
        } else if (type instanceof LongType) {
            validator = new LongValidator();
        } else if (type instanceof FloatType) {
            validator = new FloatValidator();
        } else if(type instanceof ArrayType) {
            ArrayType arrayType = (ArrayType)type; 
            type = arrayType.componentType();
            if (type instanceof DoubleType) {
                validator = new DoubleArrayValidator();
            } else if (type instanceof StringType) {
                validator = null;
            } else if (type instanceof ByteType) {
                validator = new ByteArrayValidator();
            } else if (type instanceof BooleanType) {
                validator = new BooleanArrayValidator();
            } else if (type instanceof IntegerType) {
                validator = new IntegerArrayValidator();
            } else if (type instanceof LongType) {
                validator = new LongArrayValidator();
            } else if (type instanceof FloatType) {
                validator = new FloatArrayValidator();
            }
        }
    }

    @Override
    public String isValid(String value) {
        return validator == null ? null : validator.isValid(value);
    }

    @Override
    public final void modify(WriteGraph graph, final String value) throws DatabaseException {
        VirtualGraphSupport vgs = graph.getService(VirtualGraphSupport.class);
        VirtualGraph vg = vgs.getGraph(graph, resource);
        if (vg != null) {
            graph.syncRequest(new WriteRequest(vg) {
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
                    modify0(graph, value);
                }
            });
        } else {
            modify0(graph, value);
        }
    }

    protected void modify0(WriteGraph graph, String value) throws DatabaseException {
        Layer0 l0 = Layer0.getInstance(graph);

        try {
            // FIXME: no support for StringArray!
            Datatype type = graph.getDataType(resource);
            if (type == null)
                return;

            if (type instanceof StringType) {
                if(Commands.isRecording())
                    Commands.record(graph, new ClaimValueCommand(resource, value, Bindings.STRING));

                // We do not want to parse string types with databoard since it requires the string to be within quotes "".
                graph.claimValue(resource, value, Bindings.STRING);
                return;
            }

//            } else if (type instanceof DoubleType) {
//            	double v = PrimitiveValueParser.parseDouble(value);
//            	if(converter != null) v = converter.convert(v);
//                graph.claimValue(resource, v, Bindings.DOUBLE);
//            } else if (type instanceof ByteType) {
//                graph.claimValue(resource, PrimitiveValueParser.parseByte(value), Bindings.BYTE);
//            } else if (type instanceof BooleanType) {
//                graph.claimValue(resource, PrimitiveValueParser.parseBoolean(value), Bindings.BOOLEAN);
//            } else if (type instanceof IntegerType) {
//                graph.claimValue(resource, PrimitiveValueParser.parseInt(value), Bindings.INTEGER);
//            } else if (type instanceof LongType) {
//                graph.claimValue(resource, PrimitiveValueParser.parseLong(value), Bindings.LONG);
//            } else if (type instanceof FloatType) {
//                graph.claimValue(resource, PrimitiveValueParser.parseFloat(value), Bindings.FLOAT);
//            } else if (graph.getPossibleRelatedValue(resource, l0.HasName) != null) {
//                graph.claimLiteral(resource, l0.HasName, value, Bindings.STRING);

            if (type instanceof ArrayType) {
                ArrayType arrayType = (ArrayType)type; 
                type = arrayType.componentType();
                value = "[" + value + "]";
//            if (type instanceof BooleanType) {
//            	graph.claimValue(resource, PrimitiveValueParser.parseBooleanArray(value), Bindings.BOOLEAN_ARRAY);
//            } else if (type instanceof IntegerType) {
//            	graph.claimValue(resource, PrimitiveValueParser.parseIntArray(value), Bindings.INT_ARRAY);
//            } else if (type instanceof LongType) {
//            	graph.claimValue(resource, PrimitiveValueParser.parseLongArray(value), Bindings.LONG_ARRAY);
//            } else if (type instanceof FloatType) {
//            	graph.claimValue(resource, PrimitiveValueParser.parseFloatArray(value), Bindings.FLOAT_ARRAY);
//            } else if (type instanceof DoubleType) {
////            	graph.claimValue(resource, PrimitiveValueParser.parseDoubleArray(value), Bindings.DOUBLE_ARRAY);
////                Datatype dt = graph.getPossibleRelatedValue(resource, l0.HasDataType, Bindings.getBindingUnchecked(Datatype.class));
////                Binding binding = Bindings.getBinding(dt);
////                Object parsedValue = binding.parseValue("["+value+"]", new DataValueRepository());
////                graph.claimValue(resource, parsedValue, binding);
//            } else if (type instanceof ByteType) {
//            	graph.claimValue(resource, PrimitiveValueParser.parseByteArray(value), Bindings.BYTE_ARRAY);
//            } else if (type instanceof StringType) {
//                Datatype dt = graph.getPossibleRelatedValue(resource, l0.HasDataType, Bindings.getBindingUnchecked(Datatype.class));
//                Binding binding = Bindings.getBinding(dt);
//                Object parsedValue = binding.parseValue(value, new DataValueRepository());
//                graph.claimValue(resource, parsedValue, binding);
//            }
            }

            Datatype dt = graph.getPossibleRelatedValue(resource, l0.HasDataType, Bindings.getBindingUnchecked(Datatype.class));
            Binding binding = Bindings.getBinding(dt);
            Object parsedValue = binding.parseValue(value, new DataValueRepository());

            if(Commands.isRecording())
                Commands.record(graph, new ClaimValueCommand(resource, parsedValue, binding));

            graph.claimValue(resource, parsedValue, binding);
        } catch(IllegalArgumentException e) {
            // value could not be modified
        } catch (DataTypeSyntaxError e) {
            throw new ValidationException("Could not modify resource '" + resource.getResourceId() + "' with value " + value, e);
        } catch (BindingException e) {
            throw new ValidationException("Could not modify resource '" + resource.getResourceId() + "' with value " + value, e);
        }
    }

}
