/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy  - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.Platform;
import org.osgi.framework.Bundle;
import org.osgi.framework.wiring.BundleWiring;
import org.simantics.db.ObjectResourceIdMap;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.ClusterBase;
import org.simantics.db.impl.ClusterSupport;
import org.simantics.db.impl.ClusterTraitsBase;
import org.simantics.db.impl.ResourceImpl;
import org.simantics.db.impl.support.ResourceSupport;
import org.simantics.db.request.QueryDeserializer;
import org.simantics.db.request.QueryFactory;
import org.simantics.db.request.QueryFactoryKey;
import org.simantics.db.request.Read;
import org.simantics.db.service.Bytes;
import org.slf4j.LoggerFactory;

import gnu.trove.map.hash.TIntLongHashMap;

public class QueryDeserializerImpl implements QueryDeserializer {

    private static final org.slf4j.Logger LOGGER = LoggerFactory.getLogger(QueryDeserializerImpl.class);

    QueryListening listening;
    ResourceSupport resourceSupport;
    QueryCache qc;
    QuerySupport qs;
    ClusterSupport cs;
    CacheEntry entry;
    
    private byte[] bytes;
    private int byteIndex;
    
    private TIntLongHashMap clusterKeys = new TIntLongHashMap();
    private Map<Integer,QueryFactory> ids = new HashMap<Integer,QueryFactory>();

    public QueryDeserializerImpl(QueryProcessor qp, byte[] bytes) {
        this.listening = qp.listening;
        this.resourceSupport = qp.resourceSupport;
        this.qc = qp.cache;
        this.qs = qp.querySupport;
        this.cs = qs.getClusterSupport();;
        this.bytes = bytes;
    }
    
    public byte readByte() {
        return bytes[byteIndex++];
    }

    public boolean peekZero() {
    	return bytes[byteIndex] == 0 && bytes[byteIndex+1] == 0;
    }
    
    public int readLE2() {
        int result = Bytes.readLE2(bytes, byteIndex);
        byteIndex += 2;
        return result;
    }

    public int readLE4() {
        int result = Bytes.readLE4(bytes, byteIndex);
        byteIndex += 4;
        return result;
    }
    
    public long readLE8() {
        long result = Bytes.readLE8(bytes, byteIndex);
        byteIndex += 8;
        return result;
    }

    public byte[] readBytes(int len) {
        byte[] result = Arrays.copyOfRange(bytes, byteIndex, byteIndex+len);
        byteIndex += len;
        return result;
    }
    
    public void readHeaders() {
        int idsSize = readLE4();
        for(int i=0;i<idsSize;i++) {
            int size = readLE4();
            QueryFactoryKey id = QueryFactoryKey.read(new String(readBytes(size)));
            int key = readLE4();
            try {
                Bundle bundle = Platform.getBundle(id.getBundle());
                BundleWiring bundleWiring = bundle.adapt(BundleWiring.class);
                ClassLoader classLoader = bundleWiring.getClassLoader();
                Class<QueryFactory> clazz = (Class<QueryFactory>)classLoader.loadClass(id.classId() + "Factory");
                QueryFactory qf = clazz.getDeclaredConstructor().newInstance(); 
                ids.put(key, qf);
            } catch (ClassNotFoundException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            } catch (InstantiationException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            } catch (IllegalAccessException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            } catch (IllegalArgumentException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            } catch (InvocationTargetException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            } catch (NoSuchMethodException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            } catch (SecurityException e) {
                LOGGER.error("Error while resolving QueryFactory", e);
            }
        }
        int clusterKeysSize = readLE4();
        for(int i=0;i<clusterKeysSize;i++) {
            long cluster = readLE8();
            int key = readLE4();
            clusterKeys.put(key, cluster);
        }
    }
    
    public QueryFactory readFactory() {
        int key = readLE4();
        return ids.get(key);
    }

    public int readQueries() {
        int count = readLE4();
        for(int i=0;i<count;i++) {
            QueryFactory qf = readFactory();
            try {
                qf.read(this);
            } catch (DatabaseException e) {
                e.printStackTrace();
            }
        }
        return count;
    }

    public Resource readResource() throws DatabaseException {
    	return new ResourceImpl(resourceSupport, readResourceI());
    }
    
    public int readResourceI() throws DatabaseException {
        if(peekZero()) {
            return readLE2();
        }
        int key = readLE4();
        if(key < 0)
            return key;
        int clusterKey = ClusterTraitsBase.getClusterKeyFromResourceKey(key);
        long cluster = clusterKeys.get(clusterKey);
        ClusterBase cb = cs.getClusterByClusterId(cluster);
        if(cb == null)
            return 0;
        return ClusterTraitsBase.createResourceKey(cb.getClusterKey(), ClusterTraitsBase.getResourceIndexFromResourceKey(key));
    }
    
    public byte[] readByteArray() {
        int len = readLE4();
        if(len == -1)
            return null;
        return readBytes(len);
    }
    
    public String readString() {
        return new String(readByteArray());
    }
    
    public ObjectResourceIdMap<String> createChildMap() {
        return qs.createChildMap();
    }

    AssertedPredicates readAssertedPredicates() throws DatabaseException {
        int r = readResourceI();
        return qc.getOrCreateAssertedPredicates(r);
    }

    AssertedStatements readAssertedStatements() throws DatabaseException {
        int r1 = readResourceI();
        int r2 = readResourceI();
        return qc.getOrCreateAssertedStatements(r1, r2);
    }

    ChildMap readChildMap() throws DatabaseException {
        int r = readResourceI();
        return qc.getOrCreateChildMap(r);
    }

    DirectObjects readDirectObjects() throws DatabaseException {
        int r1 = readResourceI();
        int r2 = readResourceI();
        return qc.getOrCreateDirectObjects(r1, r2);
    }
    
    DirectPredicates readDirectPredicates() throws DatabaseException {
        int r = readResourceI();
        return qc.getOrCreateDirectPredicates(r);
    }

    Objects readObjects() throws DatabaseException {
        int r1 = readResourceI();
        int r2 = readResourceI();
        return qc.getOrCreateObjects(r1, r2);
    }

    OrderedSet readOrderedSet() throws DatabaseException {
        int r = readResourceI();
        return qc.getOrCreateOrderedSet(r);
    }

    Predicates readPredicates() throws DatabaseException {
        int r = readResourceI();
        return qc.getOrCreatePredicates(r);
    }

    PrincipalTypes readPrincipalTypes() throws DatabaseException {
        int r = readResourceI();
        return qc.getOrCreatePrincipalTypes(r);
    }

    RelationInfoQuery readRelationInfoQuery() throws DatabaseException {
        int r = readResourceI();
        return qc.getOrCreateRelationInfoQuery(r);
    }

    Statements readStatements() throws DatabaseException {
        int r1 = readResourceI();
        int r2 = readResourceI();
        return qc.getOrCreateStatements(r1, r2);
    }

    SuperRelations readSuperRelations() throws DatabaseException {
        int r = readResourceI();
        return qc.getOrCreateSuperRelations(r);
    }

    SuperTypes readSuperTypes() throws DatabaseException {
        int r = readResourceI();
        return qc.getOrCreateSuperTypes(r);
    }

    TypeHierarchy readTypeHierarchy() throws DatabaseException {
        int r = readResourceI();
        return qc.getOrCreateTypeHierarchy(r);
    }

    Types readTypes() throws DatabaseException {
        int r = readResourceI();
        return qc.getOrCreateTypes(r);
    }

    URIToResource readURIToResource() throws DatabaseException {
        String s = readString();
        return qc.getOrCreateURIToResource(s);
    }

    ValueQuery readValueQuery() throws DatabaseException {
        int r = readResourceI();
        return qc.getOrCreateValueQuery(r);
    }
    
    public void setEntry(CacheEntry entry) {
    	this.entry = entry;
    }

    public CacheEntry getEntry() {
    	return entry;
    }

	@Override
	public void store(Read read, Object value) {
    	ReadEntry entry = qc.getOrCreateReadEntry(read);
    	entry.setResult(value);
    	entry.setReady();
		setEntry(entry);
	}

	@Override
	public void parent(Read read) {
		ReadEntry entry = qc.getOrCreateReadEntry(read);
		listening.addParent(getEntry(), entry);
	}
    

}
