/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package org.simantics.db.impl.query;

import org.simantics.databoard.Bindings;
import org.simantics.db.DevelopmentKeys;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.request.QueryFactoryKey;
import org.simantics.utils.Development;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class CacheEntryBase<Procedure> extends CacheEntry<Procedure> {

	private static final Logger LOGGER = LoggerFactory.getLogger(CacheEntryBase.class);

	// Default level is something that is not quite a prospect but still allows for ordering within CacheCollectionResult 
	public static final short UNDEFINED_LEVEL = 5;

	public short level = UNDEFINED_LEVEL;

	final public static CacheEntryBase[] NONE = new CacheEntryBase[0];

	static Object NO_RESULT = new Object() { public String toString() { return "NO_RESULT"; }};
	static protected Object INVALID_RESULT = new Object() { public String toString() { return "INVALID_RESULT"; }};

	//	// Just created
	//    static protected Object FRESH = new Object() { public String toString() { return "CREATED"; }};
	// Result is computed - no exception
	static protected Object READY = new Object() { public String toString() { return "READY"; }};
	// Computation is under way
	static protected Object PENDING = new Object() { public String toString() { return "PENDING"; }};
	// Entry is discarded and is waiting for garbage collect
	static protected Object DISCARDED = new Object() { public String toString() { return "DISCARDED"; }};
	// The result has been invalidated
	static protected Object REQUIRES_COMPUTATION = new Object() { public String toString() { return "REFUTED"; }};
	// The computation has excepted - the exception is in the result
	static protected Object EXCEPTED = new Object() { public String toString() { return "EXCEPTED"; }};

	// This indicates the status of the entry
	public Object statusOrException = REQUIRES_COMPUTATION;

	private int hash = 0;

	@Override
	final public int hashCode() {
		if(hash == 0) hash = makeHash();
		return hash;
	}

	abstract int makeHash();

	// This can be tested to see if the result is finished
	Object result = NO_RESULT;

	final public boolean isFresh() {
		return REQUIRES_COMPUTATION == statusOrException;
	}

	public void setReady() {
		assert(result != NO_RESULT);
		statusOrException = READY;
	}

	@Deprecated
	final public boolean isReady() {
		return READY == statusOrException || EXCEPTED == statusOrException;
	}

	@Override
	public void discard() {
		if (Development.DEVELOPMENT) {
			if(Development.<Boolean>getProperty(DevelopmentKeys.CACHE_ENTRY_STATE, Bindings.BOOLEAN)) {
				System.err.println("[QUERY STATE]: discarded " + this);
			}
		}
		statusOrException = DISCARDED;
		result = NO_RESULT;
	}

	@Override
	final public boolean isDiscarded() {
		return DISCARDED == statusOrException;
	}

	@Override
	public void refute() {
		if (Development.DEVELOPMENT) {
			if(Development.<Boolean>getProperty(DevelopmentKeys.CACHE_ENTRY_STATE, Bindings.BOOLEAN)) {
				System.err.println("[QUERY STATE]: refuted " + this);
			}
		}
		statusOrException = REQUIRES_COMPUTATION;
	}

	@Override
	final public boolean isRefuted() {
		return REQUIRES_COMPUTATION == statusOrException;
	}

	@Override
	public void except(Throwable throwable) {
		if (Development.DEVELOPMENT) {
			if(Development.<Boolean>getProperty(DevelopmentKeys.CACHE_ENTRY_STATE, Bindings.BOOLEAN)) {
				System.err.println("[QUERY STATE]: excepted " + this);
			}
		}
		if(statusOrException != DISCARDED) {
			statusOrException = EXCEPTED;
			result = throwable;
		} else {
			LOGGER.warn("Cache entry got excepted status after being discarded: " + getClass().getSimpleName(), throwable);
			result = throwable;
		}
	}

	final public void checkAndThrow() throws DatabaseException {
		if(isExcepted()) {
			Throwable throwable = (Throwable)result;
			if(throwable instanceof DatabaseException) throw (DatabaseException)throwable;
			else throw new DatabaseException(throwable);
		}
	}

	@Override
	final public boolean isExcepted() {
		return EXCEPTED == statusOrException;
	}

	@Override
	public void setPending(QuerySupport querySupport) {
		statusOrException = PENDING;
		clearResult(querySupport);
	}

	@Override
	final public boolean isPending() {
		return PENDING == statusOrException;
	}

	final public boolean requiresComputation() {
		return REQUIRES_COMPUTATION == statusOrException;
	}

	final public boolean assertPending() {
		boolean result = isPending();
		if(!result) {
			LOGGER.warn("Assertion failed, expected pending, got " + statusOrException);
		}
		return result;
	}

	final public boolean assertNotPending() {
		boolean result = !isPending();
		if(!result) {
			new Exception(this +  ": Assertion failed, expected not pending, got " + statusOrException).printStackTrace();
		}
		return result;
	}

	final public boolean assertNotDiscarded() {
		boolean result = !isDiscarded();
		if(!result) {
			new Exception(this +  ": Assertion failed, expected not discarded, got " + statusOrException).printStackTrace();
		}
		return result;
	}

	@Override
	public void setResult(Object result) {
		this.result = result;
	}

	@SuppressWarnings("unchecked")
	@Override
	final public <T> T getResult() {
		assert(statusOrException != DISCARDED);
		return (T)result;
	}

	@Override
	public void clearResult(QuerySupport support) {
		setResult(NO_RESULT);
	}

	protected String internalError() {
		return toString() + " " + statusOrException + " " + result;
	}


	protected boolean handleException(ReadGraphImpl graph, IntProcedure procedure) throws DatabaseException {
		if(isExcepted()) {
			procedure.exception(graph, (Throwable)getResult());
			return true;
		} else {
			return false;
		}
	}

	protected boolean handleException(ReadGraphImpl graph, TripleIntProcedure procedure) throws DatabaseException {
		if(isExcepted()) {
			procedure.exception(graph, (Throwable)getResult());
			return true;
		} else {
			return false;
		}
	}

	protected <T> boolean handleException(ReadGraphImpl graph, InternalProcedure<T> procedure) throws DatabaseException {
		if(isExcepted()) {
			procedure.exception(graph, (Throwable)getResult());
			return true;
		} else {
			return false;
		}
	}

	@Override
	boolean isImmutable(ReadGraphImpl graph) throws DatabaseException {
		return false;
	}

	@Override
	boolean shouldBeCollected() {
		return true;
	}

	@Override
	short getLevel() {
		return level;
	}

	@Override
	short setLevel(short level) {
		short existing = this.level;
		this.level = level;
		return existing;
	}

	@Override
	void prepareRecompute(QuerySupport querySupport) {
		setPending(querySupport);
	}

	@Override
	public Object getOriginalRequest() {
		// This is the original request for all built-in queries
		return getQuery();
	}

	public CacheEntryBase() {
	}

	public QueryFactoryKey classId() {
		return new QueryFactoryKey("org.simantics.db.impl", getClass().getName());
	}

	public void serializeKey(QuerySerializerImpl serializer) {
		throw new IllegalStateException("Cannot serialize query key for " + this);
	}

	public void serializeValue(QuerySerializerImpl serializer) {
		throw new IllegalStateException("Cannot serialize query value for " + this);
	}

	public void serializeParents(QuerySerializerImpl serializer) {
		Iterable<CacheEntry> ps = serializer.getQueryProcessor().listening.getParents(this);
		int sizePos = serializer.writeUnknownSize();
		int actual = 0;
		for(CacheEntry<?> entry : ps) {
			CacheEntryBase b = (CacheEntryBase)entry;
			if(entry.isDiscarded())
				continue;
			QueryFactoryKey cid = b.classId();
			if(cid == null) 
				continue;
			serializer.serializeId(cid);
			b.serializeKey(serializer);
			actual++;
		}
		serializer.setUnknownSize(sizePos, actual);
	}

	public long cluster(QuerySerializerImpl serializer) {
		throw new IllegalStateException("Cannot compute query cluster for " + this);
	}

	public void serialize(QuerySerializerImpl serializer) {
		serializer.serializeId(classId());
		serializeKey(serializer);
		serializeValue(serializer);
		serializeParents(serializer);
	}

}
