/*******************************************************************************
 * Copyright (c) 2007, 2024 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy  - improvements
 *******************************************************************************/
package org.simantics.db.impl.query;

import org.simantics.db.RelationInfo;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.IntProcedureAdapter;


final public class AssertedPredicates extends UnaryQuery<IntProcedure> {
	
    public AssertedPredicates(final int r) {
        super(r);
    }

    public static AssertedPredicates newInstance(final int r) {
        return new AssertedPredicates(r);
    }

    @Override
    final public void clearResult(QuerySupport support) {
        setResult(new IntArray());
    }
    
    @Override
    final public void setReady() {
        super.setReady();
        IntArray v = (IntArray)getResult();
        int size = v.size();
        if(size == 0) setResult(IntArray.EMPTY);
        else v.trim();
    }
    
 	@Override
 	final public void removeEntry(QueryProcessor provider) {
 	    provider.cache.remove(this);
 	}
 	
 	void computeInheritedAssertions(ReadGraphImpl graph, int type, final IntProcedure proc) throws DatabaseException {
 		
 		QueryProcessor processor = graph.processor;

 		QueryCache.runnerDirectObjects(graph, type, processor.getInherits(), this, null, new SyncIntProcedure() {

            @Override
            public void run(ReadGraphImpl graph) {
            }

            @Override
            public void execute(ReadGraphImpl graph,int inh) throws DatabaseException {

            	inc();
                
                QueryCache.runnerAssertedPredicates(graph, inh, AssertedPredicates.this, null, new IntProcedure() {

                    @Override
                    public void execute(ReadGraphImpl graph, int ass) {
                    	addOrSet(ass);
                    }

                    @Override
                    public void finished(ReadGraphImpl graph) throws DatabaseException {
                        dec(graph);
                    }
        			
        			@Override
        			public void exception(ReadGraphImpl graph, Throwable t) throws DatabaseException {
                        dec(graph);
                    }
                    
                });
                
            }

            @Override
            public void finished(ReadGraphImpl graph) throws DatabaseException {
                dec(graph);
            }
            
        });

 	}

    //@Override
    public Object compute(ReadGraphImpl graph, final IntProcedure proc) throws DatabaseException {

    	QueryProcessor processor = graph.processor;
    	
        computeInheritedAssertions(graph, id, proc);

    	QueryCache.runnerDirectObjects(graph, id, processor.getAsserts(), this, null, new IntProcedure() {

            @Override
            public void execute(ReadGraphImpl graph, final int ass) throws DatabaseException {
                
            	QueryCache.runnerDirectObjects(graph, ass, processor.getHasPredicate(), AssertedPredicates.this, null, new IntProcedure() {

                    @Override
                    public void execute(ReadGraphImpl graph, final int pred) throws DatabaseException {

                    	addOrSetHiding(graph, pred, AssertedPredicates.this);
                        return;
                        
                    }

                    @Override
                    public void finished(ReadGraphImpl graph) {
                    }
        			
        			@Override
        			public void exception(ReadGraphImpl graph, Throwable t) {
//        				proc.exception(graph, t);
                    }

                });
                
            }

            @Override
            public void finished(ReadGraphImpl graph) {
            }

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
//				proc.exception(graph, t);
			}
            
        });
    	
        finish(graph, processor);

    	performFromCache(graph, proc);
        
        return getResult();
        
    }
    
    @Override
    public String toString() {
    	return "AssertedPredicates[" + id + "]";
    }
    
    final public void finish(ReadGraphImpl graph, QueryProcessor provider) {
        
    	assert(!isReady());

        synchronized(this) {
        	setReady();
        }

    }

    synchronized private void addOrSet(int add) {

    	assert(isPending());
    	
        IntArray value = (IntArray)getResult(); 
        value.add(add);
        
    }

	synchronized private void addOrSetHiding(ReadGraphImpl graph, int add, CacheEntry parent) throws DatabaseException {

    	assert(isPending());
    	
        IntArray value = (IntArray)getResult(); 
		RelationInfo ri = QueryCacheBase.resultRelationInfoQuery(graph, add, parent, null);
		if(ri.isFunctional) {
			// Replace existing functional predicate if found
			try {
				IntSet supers = QueryCache.resultSuperRelations(graph, add, parent, null); 
		        if(value.data == null) {
		            if(value.sizeOrData != IntArray.NO_DATA) {
		            	if(supers.contains(value.sizeOrData)) {
		            		value.sizeOrData = add;
		            		return;
		            	}
		            }
		        } else {
		            for(int i = 0;i < value.sizeOrData ; i++) {
		            	if(supers.contains(value.data[i])) {
		            		value.data[i] = add;
		            		return;
		            	}
		            }
		        }
			} catch (Throwable e) {
				except(e);
				return;
			}
		}
	        
	    // No replacement - append
        value.add(add);

    }
    
    @Override
    public Object performFromCache(ReadGraphImpl graph, final IntProcedure procedure) throws DatabaseException {

    	assert(isReady());
    	
    	if(handleException(graph, procedure)) return EXCEPTED;
    	
        final IntArray value = (IntArray)getResult();
        if(value.data == null) {
            if(value.sizeOrData != IntArray.NO_DATA) procedure.execute(graph, value.sizeOrData);
        } else {
            for(int i = 0;i < value.sizeOrData ; i++) procedure.execute(graph, value.data[i]);
        }

        procedure.finished(graph);
        
        return value;
        
    }
    
    @Override
    public void recompute(ReadGraphImpl graph) throws DatabaseException {

        compute(graph, new IntProcedureAdapter() {

            @Override
            public void finished(ReadGraphImpl graph) {
            }

            @Override
            public void exception(ReadGraphImpl graph, Throwable t) {
                new Error("Error in recompute.", t).printStackTrace();
            }

        });
        
    }

    @Override
    public void serializeValue(QuerySerializerImpl serializer) {
        IntArray is = getResult();
        is.serialize(serializer);
    }

}
