package org.simantics.db.impl.graph;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.simantics.db.impl.BlockingAsyncProcedure;
import org.simantics.db.impl.query.CacheEntry;
import org.simantics.db.impl.query.QueryProcessor.SessionTask;

public class BarrierTracing {

	public static final boolean BOOKKEEPING = false;
	static final boolean RESTART_GUARD = BOOKKEEPING;

	public static Map<SessionTask,Exception> tasks = new HashMap<>();
	public static final HashMap<AsyncBarrierImpl, Collection<AsyncBarrierImpl>> reverseLookup = new HashMap<>();
	public static final HashMap<AsyncBarrierImpl, Debugger> debuggerMap = new HashMap<>();
	public static final HashMap<AsyncBarrierImpl, Throwable> restartMap = new HashMap<>();
	public static final HashMap<AsyncBarrierImpl, Throwable> startMap = new HashMap<>();
	public static final HashMap<BlockingAsyncProcedure, Throwable> baps = new HashMap<>();

	synchronized public static void registerBAP(BlockingAsyncProcedure bap) {
		baps.put(bap, new Exception());
	}

	synchronized public static void unregisterBAP(BlockingAsyncProcedure bap) {
		baps.remove(bap);
	}

	synchronized public static void printBAPS() {
		for(BlockingAsyncProcedure bap : baps.keySet()) {
			bap.print();
			Throwable t = baps.get(bap);
			if(t != null)
				t.printStackTrace();
		}
	}

	synchronized public static void report() {
		for(Map.Entry<AsyncBarrierImpl, Debugger> entry : debuggerMap.entrySet()) {
			AsyncBarrierImpl barrier = entry.getKey();
			if(barrier.get() != 0) {
				Debugger d = entry.getValue();
				d.print();
			}
		}
	}

	public static void trace(AsyncBarrierImpl barrier, CacheEntry<?> entry) {

		if (RESTART_GUARD) {
			synchronized (startMap) {
				startMap.put(barrier, new Exception());
			}
		}
		synchronized (debuggerMap) {
			debuggerMap.put(barrier, new Debugger(entry));
		}
		synchronized (reverseLookup) {
			Collection<AsyncBarrierImpl> barriers = reverseLookup
					.get(barrier.caller);
			if (barriers == null) {
				barriers = new ArrayList<AsyncBarrierImpl>();
				reverseLookup.put(barrier.caller, barriers);
			}
			barriers.add(barrier);
		}

	}

	public static void inc(AsyncBarrierImpl barrier, int count, Object id) {

		synchronized (debuggerMap) {
			Debugger debugger = debuggerMap.get(barrier);
			debugger.inc(System.identityHashCode(barrier) + " - " + id + " => " + count);
		}

		if (RESTART_GUARD)
			if(restartMap.containsKey(barrier)) {
				startMap.get(barrier).printStackTrace();
				restartMap.get(barrier).printStackTrace();
				new Exception().printStackTrace();
				throw new IllegalStateException("Unplanned restart");
			}


	}

	public static void dec(AsyncBarrierImpl barrier, int count, Object id) {

		synchronized (debuggerMap) {
			Debugger debugger = debuggerMap.get(barrier);
			debugger.dec(System.identityHashCode(barrier) + " - " + id + " => " + count);
		}

		if (count == 0) {
			if (RESTART_GUARD) {
				restartMap.put(barrier, new Exception());
			}
			//debuggerMap.remove(barrier);
		}
		else if (count < 0) {

			synchronized (debuggerMap) {
				Debugger debugger = debuggerMap.get(barrier);
				debugger.print();
			}

			BarrierTracing.startMap.get(barrier).printStackTrace();
			BarrierTracing.restartMap.get(barrier).printStackTrace();
			new Exception().printStackTrace();
		}
	}

	public static class Debugger {

		Object entry;
		List<Exception> exceptions = new ArrayList<>();

		Debugger(Object entry) {
			this.entry = entry;
			exceptions.add(new Exception());
		}

		public synchronized void inc(String message) {
			exceptions.add(new Exception(message));
		}

		public synchronized void dec(String message) {
			exceptions.add(new Exception(message));
		}

		public synchronized void print() {
			System.err.println("Printing barrier invocation log with " + exceptions.size() + " elements " + entry);
			exceptions.add(new Exception(" = DONE ="));
			for (Exception e : exceptions) {
				e.printStackTrace();
			}
		}

	}

}
