/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.util.binary;

import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;

import org.simantics.databoard.util.StreamUtil;

/**
 * Input stream reader
 * 
 * @author Toni Kalajainen (toni.kalajainen@vtt.fi)
 */
public class InputStreamReadable implements BinaryReadable {

	InputStream is;
	long limit, position;
	
	public static BinaryReadable readFully(InputStream is) throws IOException {
		byte[] data = StreamUtil.readFully(is);
		return new BinaryMemory( data );
	}
	
	public InputStreamReadable(InputStream is, long limit)
	{
		this.is = is;
		this.limit = limit;
	}

	/**
	 * Get next byte
	 * @return 0..255
	 * @throws IOException
	 */
	int _get()
	throws IOException
	{
		int value = is.read();
		if (value==-1)
			throw new EOFException();
		position++;
		return value & 0xff;
	}
	
	/**
	 * Get next byte
	 * @return 0..255 or -1 on end of file
	 * @throws IOException
	 */
	int _read()
	throws IOException
	{
		int value = is.read();
		if (value==-1) return -1;
		position++;		
		return value & 0xff;
	}	
	
    public final String readLine() throws IOException {
    	StringBuffer input = new StringBuffer();
    	int c = -1;
    	boolean eol = false;

    	while (!eol) {
    	    switch (c = _read()) {
    	    case -1:
    	    case '\n':
    		eol = true;
    		break;
    	    case '\r':
    		eol = true;
//    		long cur = position();
//    		if ((_read()) != '\n') {
//    		    position=cur;
//    		}
    		break;
    	    default:
    		input.append((char)c);
    		break;
    	    }
    	}

    	if ((c == -1) && (input.length() == 0)) {
    	    return null;
    	}
    	return input.toString();
    }	
	
	
	@Override
	public byte readByte() 
    throws IOException	
	{
		return (byte) _get();
	}
	
	@Override
	public char readChar() throws IOException {
		return (char) ( (_get() << 8) |  _get() ) ;
	}
	
	@Override
	public int readUnsignedByte() throws IOException {
		return _get() & 0x000000ff;
	}	

	@Override
	public boolean readBoolean() 
    throws IOException	
	{
		return _get()!=0;
	}	

	@Override
	public void readFully(byte[] dst, int offset, int length) 
    throws IOException	
	{
		while (length>0) {
			int bytesRead = is.read(dst, offset, length);
			if (bytesRead==-1) throw new EOFException();
			position+=bytesRead;			
			offset += bytesRead;
			length -= bytesRead;
		}
	}

	@Override
	public void readFully(byte[] dst) 
    throws IOException	
	{
		readFully(dst, 0, dst.length);
	}

	@Override
	public void readFully(ByteBuffer buf) 
    throws IOException	
	{		
		for (;buf.hasRemaining();)
			buf.put((byte)_get());		
	}

	@Override
	public void readFully(ByteBuffer buf, int length) 
    throws IOException	
	{
		if (length<256) {
			for (int i=0; i<length; i++)
				buf.put((byte)_get());
		} else {
			byte[] b = new byte[length];
			readFully(b, 0, length);
			buf.put(b);
		}
	}

	@Override
	public double readDouble() 
    throws IOException	
	{
		return Double.longBitsToDouble(readLong());
	}

	@Override
	public float readFloat() 
    throws IOException	
	{
		return Float.intBitsToFloat(readInt());
	}
	
    public final String readUTF() throws IOException {
    	return DataInputStream.readUTF(this);
    } 	

	@Override
	public int readInt() 
    throws IOException	
	{
		return 
			( _get() << 24) |
			( _get() << 16) | 
			( _get() << 8) |
			( _get() );
	}

	@Override
	public long readLong() 
    throws IOException	
	{
		return
		( ((long)_get()) << 56) |
		( ((long)_get()) << 48) | 
		( ((long)_get()) << 40) |
		( ((long)_get()) << 32) |		
		( ((long)_get()) << 24) |
		( ((long)_get()) << 16) | 
		( ((long)_get()) << 8) |
		( ((long)_get()) );		
	}

	@Override
	public short readShort() 
    throws IOException	
	{
		return (short) ( (_get() << 8) |  _get() ) ;
	}

	@Override
	public int readUnsignedShort() 
    throws IOException	
	{
		return ( (_get() << 8) |  _get() ) ;
	}	
	
	@Override
	public long length() 
	{
		return limit;
	}
	
	@Override
	public long position() {
		return position;
	}


	@Override
	public long skipBytes(long bytes) throws IOException {
		return is.skip(bytes);	
	}

	@Override
	public int skipBytes(int bytes) throws IOException {
		long b = is.skip(bytes);
		if (b > Integer.MAX_VALUE) {
			throw new IOException("is.skip(" + bytes + ") returned " + b + " > Integer.MAX_VALUE");
		}
		return (int) b;
	}
	
}

