/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.util.Collection;
import java.util.Iterator;
import java.util.PriorityQueue;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.type.ArrayType;

/**
 * PriorityQueueBindings binds ArrayType to java.util.PriorityQueue 
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class PriorityQueueBinding extends ArrayBinding {

	public PriorityQueueBinding(Binding componentBinding) {
		this(new ArrayType(componentBinding.type()), componentBinding);
	}
	
	public PriorityQueueBinding(ArrayType type, Binding componentBinding) {
		super(type, componentBinding);
		if (type==null) throw new IllegalArgumentException("null arg");
		this.type = type;
	}
	
	@Override
	public Object create() {
		return new PriorityQueue<Object>(11, componentBinding);
	}
	
	@Override
	public Object create(Collection<Object> collection ) throws BindingException {
		PriorityQueue<Object> result = new PriorityQueue<Object>( collection.size(), componentBinding );
		result.addAll( collection );
		return result;
	}
	
	/**
	 * Create new ArrayList
	 */
	@Override
	public Object create(int length, Iterator<Object> values) {
		PriorityQueue<Object> result = new PriorityQueue<Object>( length, componentBinding );
		while (values.hasNext())
			result.add(values.next());
		return result;
	}

	public Object create(Object[] values) {
		PriorityQueue<Object> result = new PriorityQueue<Object>(values.length, componentBinding );
		for (int i=0; i<values.length; i++)
			result.add(values[i]);
		return result;
	}
	
	
	@SuppressWarnings("unchecked")
	@Override
	public Object get(Object queue, int index) throws BindingException {
		if (!isInstance(queue)) throw new BindingException("Unexpected class "+queue.getClass().getSimpleName()+", PriorityQueue expected");
		PriorityQueue<Object> q = (PriorityQueue<Object>) queue; 
		if ( index<0 || index>=q.size() ) throw new BindingException("Index out of bounds");
		Iterator<Object> it = q.iterator();
		Object result = null;
		for (int i=0; i<=index; i++)
		{
			result = it.next();
		}
		return result;
	}
	
	@SuppressWarnings("unchecked")
    @Override
	public void getAll(Object queue, Object[] result) throws BindingException {
		PriorityQueue<Object> q = (PriorityQueue<Object>) queue;
		int i=0;
		for (Iterator<Object> it = q.iterator(); it.hasNext();)
			result[ i++ ] = it.next();
	}
	
    @Override    
	public void set(Object array, int index, Object value) throws BindingException {
		remove(array, index, 1);
		add(array, index, value);
	}

	@SuppressWarnings("unchecked")
    @Override
	public void add(Object queue, int index, Object element) throws BindingException, IndexOutOfBoundsException {
		PriorityQueue<Object> q = (PriorityQueue<Object>) queue;
		q.add( element );
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public void remove(Object queue, int index, int count) throws BindingException {
		PriorityQueue<Object> q = (PriorityQueue<Object>) queue;
		Iterator<Object> it = q.iterator();
		if (index<0 || index>=q.size()) throw new BindingException("Index out of bounds");
		for (int i=0; i<=index; i++) it.next();
		it.remove();
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public int size(Object queue) throws BindingException {		
		if (!isInstance(queue)) throw new BindingException("Unexpected class "+queue.getClass().getSimpleName()+", PriorityQueue expected");		
		PriorityQueue<Object> q = (PriorityQueue<Object>) queue;
		return q.size();
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof PriorityQueue<?>;
	}

	@Override
	public void setSize(Object queue, int newSize) throws BindingException {
		@SuppressWarnings("unchecked")
		PriorityQueue<Object> q = (PriorityQueue<Object>) queue;
		int c = q.size() - newSize;
		for (int i=0; i<c; i++) q.remove();
	}		
	
	@Override
	public boolean isImmutable() {
		return false;
	}

	@Override
	public boolean isResizable() {
		return true;
	}
	
}

