/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.MapBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.reflection.VoidBinding;
import org.simantics.databoard.type.MapType;
import org.simantics.databoard.type.RecordType;

/**
 * Binds Databoard Map(T, {}) to java.util.Set and instantiates java.util.HashSet
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
@SuppressWarnings({"rawtypes", "unchecked"})
public class HashSetBinding extends MapBinding {
	
	public HashSetBinding(MapType mapType, Binding elementBinding) {
		super(mapType, elementBinding, VoidBinding.VOID_BINDING);
	}
	
	public HashSetBinding(Binding elementBinding) {
		super(new MapType(elementBinding.type(), RecordType.VOID_TYPE), elementBinding, VoidBinding.VOID_BINDING);
	}

	@Override
	public void clear(Object set) throws BindingException {
		Set _set = (Set) set;
		_set.clear();
	}

	@Override
	public boolean containsKey(Object set, Object key) throws BindingException {
//		HashSet _set = (HashSet) set;
//		return _set.contains(key);
		Set s = ((Set)set);
		Binding kb = getKeyBinding();
		
		for (Object v : s)
		{
			if (kb.equals(v, key)) return true;
		}
		return false;		
	}

	@Override
	public boolean containsValue(Object set, Object value)
			throws BindingException {
		return false;
	}

	@Override
	public Object create() throws BindingException {
		return new HashSet<Object>();
	}

	public Object create(Set<?> initialSet) throws BindingException {
		return new HashSet<Object>(initialSet);
	}

	@Override
	public Object create(Map initialMap) throws BindingException {
		return new HashSet<Object>(initialMap.keySet());
	}

	@Override
	public Object create(Object[] keys, Object[] values)
	throws BindingException {		
		HashSet<Object> result = new HashSet<Object>(keys.length);
		for (int i=0; i<keys.length; i++)
			result.add(keys[i]);
		return result;
	}
	
	@Override
	public Object create(List<Object> keys, List<Object> values) {
		HashSet<Object> result = new HashSet<Object>(keys.size());
		for (int i=0; i<keys.size(); i++)
			result.add(keys.get(i));
		return result;
	}
	
	@Override
	public Object get(Object set, Object key) throws BindingException {
		return null;
	}

	@Override
	public void getAll(Object setFrom, Map to) throws BindingException {
		Map _to = (Map) to;
		HashSet<Object> _setFrom = (HashSet<Object>) setFrom;
		for (Object k : _setFrom)
			_to.put(k, null);
	}

	@Override
	public void getAll(Object setFrom, Object[] keys, Object[] values)
			throws BindingException {
		Set<Object> _setFrom = (Set<Object>) setFrom;
		int i=0;
		for (Object k : _setFrom) {
			keys[i] = k;
			values[i] = null;
			i++;
		}
	}

	@Override
	public Object[] getKeys(Object set) throws BindingException {
		Set<Object> _set = (Set<Object>) set;
		Object[] result = _set.toArray(new Object[_set.size()]);
		Arrays.sort(result, getKeyBinding());		
		return result;
	}
	
	@Override
	public void getKeys(Object set, Set keys) throws BindingException {
		Set<Object> s = (Set<Object>) set;
		keys.addAll(s);
	}
	
	@Override
	public int count(Object src, Object from, boolean fromInclusive,
			Object end, boolean endInclusive) throws BindingException {
		int result = 0;
		Set<Object> m = ((Set<Object>)src);
		for (Object k : m) {
			int fk = keyBinding.compare(from, k);
			int ek = keyBinding.compare(k, end);
			boolean fromMatches = fromInclusive ? fk<=0 : fk<0;
			boolean endMatches = endInclusive ? ek<=0 : ek <0;			
			if ( fromMatches && endMatches ) result++;
		}		
		return result;
	}

	public int getEntries(Object src, Object from, boolean fromInclusive, Object end, boolean endInclusive, ArrayBinding dstKeyArrayBinding, Object dstKeyArray, ArrayBinding dstValueArrayBinding, Object dstValueArray, int limit) throws BindingException {
		return 0;
	}

	@Override
	public Object[] getValues(Object set) throws BindingException {
		Set<Object> _set = (Set<Object>) set;
		return new Object[_set.size()];
	}

	@Override
	public void put(Object set, Object key, Object value)
			throws BindingException {
		if (value!=null) throw new BindingException("Cannot put non-null to a Set");
		Set<Object> s = (Set<Object>) set;
		Binding kb = getKeyBinding();
		
		for (Object e : s)
		{
			if (kb.equals(e, key)) {
				s.remove(e);
				s.add(value);
				return;
			}
		}		
		s.add(key);
	}
	
	Object getComparableKey(Object set, Object key) {
		// if (keyIsComparable) return key;
		
		Set s = (Set) set;
		Binding kb = getKeyBinding();
		for (Object k : s)
		{
			if (kb.equals(k, key)) return k;
		}
		return key;
	}

	public void putAll(Object setTo, Set from) {
		Set<Object> _set = (Set<Object>) setTo;
		_set.addAll(from);
	}
	
	@Override
	public void putAll(Object setTo, Map from) throws BindingException {
		Set<Object> s = (Set<Object>) setTo;
		for (Entry<Object, Object> e : (Set<Entry<Object, Object>>) from.entrySet()) {
			Object k = getComparableKey(s, e.getKey());
			s.remove(k);			
			s.add(e.getKey());
		}
	}

	@Override
	public Object remove(Object set, Object key) throws BindingException {
		Set<Object> _set = (Set<Object>) set;
		_set.remove(key);
		return null;
	}

	@Override
	public int size(Object set) throws BindingException {
		Set<Object> _set = (Set<Object>) set;
		return _set.size();
	}

	@Override
	public boolean isInstance(Object obj) {
		return obj instanceof Set;
	}

	@Override
	public Object getCeilingKey(Object set, Object key) {
		Set<Object> s = (Set<Object>) set;
		if (s.isEmpty()) return null;
		Comparator<Object> comparator = getKeyBinding();
		Object pivot = null;
		for (Object o : s) {
			// We are trying to find key > o > pivot
			int c2 = comparator.compare(key, o);
			if (c2>0) continue;
			if (pivot==null) {pivot = o; continue;}
			int c1 = comparator.compare(o, pivot);
			if (c1<0) pivot = o;
		}
		return pivot;
	}

	@Override
	public Object getFirstKey(Object set) {
		Set<Object> s = (Set<Object>) set;
		if (s.isEmpty()) return null;
		Comparator<Object> c = getKeyBinding();
		Object result = null;
		for (Object o : s) {
			if (result==null) {
				result = o;
				continue;
			}
			if (c.compare(o, result)<0) result = o;
		}	
		
		return result;	
	}

	@Override
	public Object getFloorKey(Object set, Object key) {
		Set<Object> s = (Set<Object>) set;
		if (s.isEmpty()) return null;	
		Comparator<Object> comparator = getKeyBinding();
		Object pivot = null;
		for (Object o : s) {
			// We are trying to find pivot <= o <= key
			int c2 = comparator.compare(o, key);
			if (c2==0) return o;
			if (c2>0) continue;
			if (pivot==null) {pivot = o; continue;}
			int c1 = comparator.compare(pivot, o);
			if (c1<0) pivot = o;
		}
		return pivot;
	}

	@Override
	public Object getHigherKey(Object set, Object key) {
		Set<Object> s = (Set<Object>) set;
		if (s.isEmpty()) return null;
		Comparator<Object> comparator = getKeyBinding();
		Object pivot = null;
		for (Object o : s) {
			// We are trying to find key > o > pivot
			int c2 = comparator.compare(key, o);
			if (c2>=0) continue;
			if (pivot==null) {pivot = o; continue;}
			int c1 = comparator.compare(o, pivot);
			if (c1<0) pivot = o;
		}
		return pivot;
	}

	@Override
	public Object getLastKey(Object set) {
		Set<Object> s = (Set<Object>) set;
		if (s.isEmpty()) return null;
		Comparator<Object> c = getKeyBinding();
		Object result = null;
		for (Object o : s) {
			if (result==null) {
				result = o;
				continue;
			}
			if (c.compare(o, result)>0) result = o;
		}			
		return result;	
	}

	@Override
	public Object getLowerKey(Object set, Object key) {
		Set<Object> s = (Set<Object>) set;
		if (s.isEmpty()) return null;
		Comparator<Object> comparator = getKeyBinding();
		Object pivot = null;
		for (Object o : s) {
			// We are trying to find pivot < o < key
			int c2 = comparator.compare(o, key);
			if (c2>=0) continue;
			if (pivot==null) {pivot = o; continue;}
			int c1 = comparator.compare(pivot, o);
			if (c1<0) pivot = o;
		}
		return pivot;
	}
	

}

