/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.accessor.interestset;

import org.simantics.databoard.annotations.Union;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.type.BooleanType;
import org.simantics.databoard.type.ByteType;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.FloatType;
import org.simantics.databoard.type.IntegerType;
import org.simantics.databoard.type.LongType;
import org.simantics.databoard.type.MapType;
import org.simantics.databoard.type.OptionalType;
import org.simantics.databoard.type.RecordType;
import org.simantics.databoard.type.StringType;
import org.simantics.databoard.type.UnionType;
import org.simantics.databoard.type.VariantType;

/**
 * InterestSet is a description about what is monitored in a data model.
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
@Union({BooleanInterestSet.class,
		ByteInterestSet.class,
		IntegerInterestSet.class,
		LongInterestSet.class,
		FloatInterestSet.class,
		DoubleInterestSet.class,
		StringInterestSet.class,
		RecordInterestSet.class,
		ArrayInterestSet.class,
		MapInterestSet.class,
		OptionalInterestSet.class,
		UnionInterestSet.class,
	    VariantInterestSet.class
	    })
public abstract class InterestSet {

//	private static final String[] EMPTY_ARRAY = new String[0]; 
	
	/**
	 * Returns true, if interested in notifications of modifications, this question excludes
	 * an interest to the values.
	 * 
	 * @return true if interested in notifications
	 */
	public abstract boolean inNotifications();
	
	/**
	 * Returns true, if interested in notifications of changes and the new values.
	 * 
	 * @return true if interested in notifications and value
	 */
	public abstract boolean inValues();
	

	/**
	 * Create a new interest set 
	 * 
	 * @param type the type to create interest set for
	 * @param notification interested in notification of changes, not the values
	 * @param value interested in values (includes notification)
	 * @param recursive interested in values recursivelu (includes notification)
	 */
	public static InterestSet newInterestSet(Datatype type, boolean notification, boolean value, boolean recursive) {
		
		if (type instanceof BooleanType) {
			if (recursive) return BooleanInterestSet.MONITOR_EVERYTHING;
			return new BooleanInterestSet(notification, value);
		}
		
		if (type instanceof ByteType) {
			if (recursive) return ByteInterestSet.MONITOR_EVERYTHING;
			return new ByteInterestSet(notification, value);
		}
		
		if (type instanceof IntegerType) {
			if (recursive) return IntegerInterestSet.MONITOR_EVERYTHING;
			return new IntegerInterestSet(notification, value);
		}
		
		if (type instanceof LongType) {
			if (recursive) return LongInterestSet.MONITOR_EVERYTHING;
			return new LongInterestSet(notification, value);
		}
		
		if (type instanceof FloatType) {
			if (recursive) return FloatInterestSet.MONITOR_EVERYTHING;
			return new FloatInterestSet(notification, value);
		}
		
		if (type instanceof DoubleType) {
			if (recursive) return DoubleInterestSet.MONITOR_EVERYTHING;
			return new DoubleInterestSet(notification, value);
		}
		
		if (type instanceof StringType) {
			if (recursive) return StringInterestSet.MONITOR_EVERYTHING;
			return new StringInterestSet(notification, value);
		}
		
		if (type instanceof OptionalType) {
			OptionalType ot = (OptionalType) type;
			InterestSet cis = recursive ? newInterestSet(ot.getComponentType(), notification, value, true) : null;
			return new OptionalInterestSet(notification, value, cis );
		}
		
		if (type instanceof RecordType) {
			if (!recursive) {
				return new RecordInterestSet(notification, null, value, null, null);
			}
			RecordType rt = (RecordType) type;
			InterestSet[] componentInterests = new InterestSet[rt.getComponentCount()];
			for (int i=0; i<componentInterests.length; i++) {
				componentInterests[i] = newInterestSet( rt.getComponentType(i), notification, value, true );
			}
			return new RecordInterestSet(true, null, true, null, componentInterests);
		}
		
		if (type instanceof UnionType) {
			if (!recursive) {
				return new UnionInterestSet(notification, value, null);				
			}
			
			UnionType ut = (UnionType) type;			
			InterestSet[] componentInterests = new InterestSet[ut.getComponentCount()];
			for (int i=0; i<componentInterests.length; i++) {
				componentInterests[i] = newInterestSet( ut.getComponent(i).type, notification, value, true );
			}
			return new UnionInterestSet(notification, value, componentInterests);
		}
		
		if (type instanceof ArrayType) {
			ArrayType at = (ArrayType) type;
			InterestSet cis = recursive ? newInterestSet( at.componentType(), notification, value, true ) : null;
			return new ArrayInterestSet(notification, null, value, null, cis, null);
		}
		
		if (type instanceof MapType) {
			MapType mt = (MapType) type;
			InterestSet cis = recursive ? newInterestSet( mt.valueType, notification, value, true ) : null;
			return new MapInterestSet(notification, null, value, null, cis, null);
		}
		
		if (type instanceof VariantType) {
			return new VariantInterestSet(notification, value, null, recursive);
		}
		
		throw new IllegalArgumentException(type.toSingleLineString());		
	}

//	public String[] getCustomEvents() {
//		return customEvents;
//	}

//	public void setCustomEvents(String[] customEvents) {
//		this.customEvents = customEvents;
//	}
	/** A list of custom event types of interest */
//	public String[] customEvents = EMPTY_ARRAY;
	
	
}

