/*******************************************************************************
 * Copyright (c) 2007, 2025 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - issue #4262, GitLab #1227
 *******************************************************************************/
package org.simantics.charts.ui;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.simantics.Simantics;
import org.simantics.charts.Activator;
import org.simantics.charts.commandlog.NewChartGroupCommand;
import org.simantics.charts.ontology.ChartResource;
import org.simantics.databoard.Bindings;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteResultRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.ActionFactory;
import org.simantics.layer0.Layer0;
import org.simantics.utils.commandlog.Commands;

/**
 * @author Tuukka Lehtonen
 */
public class NewChartGroup implements ActionFactory {

    @Override
    public Runnable create(Object target) {
        if(!(target instanceof Resource))
            return null;
        final Resource parent = (Resource)target;

        return () -> {
            try {
                createChartGroup(parent);
            } catch (DatabaseException e) {
                Activator.getDefault().getLog().log(new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Failed to create chart group.", e)); //$NON-NLS-1$
            }
        };
    }

    public static Resource createChartGroup (final Resource target) throws DatabaseException {
        return createChartGroup(target, "Chart Group"); //$NON-NLS-1$
    }

    public static Resource createChartGroup (Resource target, String chartGroupNameProposal) throws DatabaseException {
        return Simantics.getSession().syncRequest(new WriteResultRequest<Resource>() {
            @Override
            public Resource perform(WriteGraph graph) throws DatabaseException {
                graph.markUndoPoint();
                return createChartGroup(graph, target, chartGroupNameProposal);
            }
        });
    }

    public static Resource createChartGroup (WriteGraph g, Resource target, String chartGroupNameProposal) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(g);
        ChartResource CHART = ChartResource.getInstance(g);

        String freshName = NameUtils.findFreshName(g, chartGroupNameProposal, target);

        Resource chart = g.newResource();
        g.claim(chart, L0.InstanceOf, null, CHART.ChartGroup);
        g.claimLiteral(chart, L0.HasName, freshName, Bindings.STRING);
        g.claim(chart, L0.PartOf, target);

        if(Commands.isRecording())
            Commands.record(g, new NewChartGroupCommand(chart, target));

        return chart;
    }

}
