/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.charts.query;

import java.util.UUID;

import org.simantics.charts.ontology.ChartResource;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.layer0.Layer0;
import org.simantics.trend.configuration.TrendItem;
import org.simantics.trend.configuration.TrendSpec;
import org.simantics.trend.configuration.ViewProfile;
import org.simantics.trend.configuration.YAxisMode;

/**
 * This query reads trend configuration (Resource) and returns TrendSpec.
 * 
 * @author Tuukka Lehtonen
 */
public class TrendSpecQuery extends BinaryRead<UUID, Resource, TrendSpec> {

    public TrendSpecQuery(UUID id, Resource resource) {
        super(id, resource);
    }

    @Override
    public TrendSpec perform(ReadGraph graph) throws DatabaseException {
        Resource chart = parameter2;

        ChartResource CHART = ChartResource.getInstance(graph);
        Layer0 L0 = Layer0.getInstance(graph);
        
        TrendSpec spec = new TrendSpec();
        spec.init();
        if ( !graph.hasStatement(chart) ) return spec;
        spec.name = graph.getPossibleRelatedValue(chart, L0.HasName, Bindings.STRING);
        if (spec.name == null)
            spec.name = "Unnamed Chart";

        spec.axisMode = YAxisMode.MultiAxis;
        Resource axisMode = graph.getPossibleObject(chart, CHART.Chart_YAxisMode);
        if (CHART.YAxisMode_SingleAxis.equals(axisMode))
            spec.axisMode = YAxisMode.SingleAxis;
        else if (CHART.YAxisMode_MultiAxis.equals(axisMode))
            spec.axisMode = YAxisMode.MultiAxis;
        
        readProfile(graph, chart, spec.viewProfile);

        for (Resource plot : graph.syncRequest(new ObjectsWithType(chart, L0.ConsistsOf, CHART.Chart_Item))) {
        	TrendItem item = graph.sync( new TrendItemQuery(plot) );
//        	if ( item.subscriptionId.equals("") ) continue;
            spec.items.add( item );
        }

        spec.sortItems();
        return spec;
    }

    private void readProfile(ReadGraph graph, Resource profile, ViewProfile viewProfile) throws DatabaseException {
        viewProfile.profileName = "Custom Profile";
        viewProfile.showMilestones = false;
//        viewProfile.timeWindow.timeWindowStart = 0.0;
//        viewProfile.timeWindow.timeWindowLength = 60.0;
        viewProfile.timeWindow.timeWindowIncrement = 50.0;

        if (profile != null) {
            Layer0 L0 = Layer0.getInstance(graph);
            ChartResource CHART = ChartResource.getInstance(graph);
            String pn = graph.getPossibleRelatedValue(profile, L0.HasName);
            if (pn != null)
                viewProfile.profileName = pn;

            viewProfile.showMilestones = Boolean.TRUE.equals( graph.getPossibleRelatedValue(profile, CHART.Chart_ShowMilestones, Bindings.BOOLEAN) );
            viewProfile.showGrid       = Boolean.TRUE.equals( graph.getPossibleRelatedValue(profile, CHART.Chart_showGrid, Bindings.BOOLEAN) );

            Double windowStart = graph.getPossibleRelatedAdapter(profile, CHART.Chart_TimeWindowStart, Double.class);
            if (windowStart != null && !Double.isNaN(windowStart))
                viewProfile.timeWindow.timeWindowStart = windowStart;
            Double windowLength = graph.getPossibleRelatedAdapter(profile, CHART.Chart_TimeWindowLength, Double.class);
            if (windowLength != null)
                viewProfile.timeWindow.timeWindowLength = windowLength;
            Double windowIncrement = graph.getPossibleRelatedAdapter(profile, CHART.Chart_TimeWindowIncrement, Double.class);
            if (windowIncrement != null)
                viewProfile.timeWindow.timeWindowIncrement = windowIncrement;

            Boolean trackExperimentTime = graph.getPossibleRelatedValue(profile, CHART.Chart_trackExperimentTime, Bindings.BOOLEAN);
            if (trackExperimentTime != null)
                viewProfile.trackExperimentTime = trackExperimentTime;
            double[] valueViewPosition = graph.getPossibleRelatedValue(profile, CHART.Chart_valueViewPosition, Bindings.DOUBLE_ARRAY);
            if (valueViewPosition != null && valueViewPosition.length >= 2) {
                viewProfile.valueViewPositionX = valueViewPosition[0];
                viewProfile.valueViewPositionY = valueViewPosition[1];
            }

            float[] bgColor = graph.getPossibleRelatedValue(profile, CHART.Chart_backgroundColor, Bindings.FLOAT_ARRAY);
            if (bgColor != null && bgColor.length >= 3) {
                viewProfile.backgroundColor = bgColor;
            }

            float[] gridColor = graph.getPossibleRelatedValue(profile, CHART.Chart_gridColor, Bindings.FLOAT_ARRAY);
            if (gridColor != null && gridColor.length >= 3) {
                viewProfile.gridColor = gridColor;
            }
        }
    }

}