/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common.processors;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.NodeContext.QueryKey;
import org.simantics.browsing.ui.NodeQueryManager;
import org.simantics.browsing.ui.Tester;
import org.simantics.browsing.ui.common.EvaluatorData;
import org.simantics.browsing.ui.common.EvaluatorData.Evaluator;
import org.simantics.browsing.ui.common.EvaluatorData.EvaluatorTree;
import org.simantics.browsing.ui.common.Preference;
import org.simantics.utils.datastructures.collections.CollectionUtils;

/**
 * A base class for query processors that resolve a set of suitable factory
 * classes of a kind for a given INodeContext.
 * 
 * <p>
 * The purpose of this class is to generically implement a basic logic for
 * resolving available factories (viewpoint/labeler/labeldecorator/comparator)
 * any given input. The implementation works based on an
 * <code>EvaluatorData</code> instance that is specified externally.The logic is
 * as follows:
 * </p>
 * <ol>
 * <li>Get all possible <code>Evaluator</code>s from the externally specified
 * EvaluatorData (see {@link EvaluatorData#get(Object)}).</li>
 * <li>For each <code>Evaluator</code>:
 * <ul>
 * <li>Walk the <code>EvaluatorTree</code> structure returned by
 * {@link #getEvaluatorTree(Evaluator)} while checking at each node whether the
 * {@link Tester} returned by {@link EvaluatorTree#getTester()} returns
 * <code>true</code>. If <code>true</code> is returned, the factories at this
 * node ({@link EvaluatorTree#getAcceptedFactories()}) are added to the result
 * and all children ({@link EvaluatorTree#getChildren()}) are recursively
 * walked. Otherwise if <code>false</code> is returned, the walking of the
 * subtree will end.</li>
 * </ul>
 * </li>
 * <li>Sort the results in descending preference order, larger number equals
 * higher preference</li>
 * </ol>
 * 
 * <p>
 * This class intended for sub-classing, please implement these methods:
 * <ul>
 * <li><code>getEvaluatorTree(Evaluator)</code> - return the appropriate
 * <code>EvaluatorTree</code> of the specified <code>Evaluator</code></li>
 * </ul>
 * 
 * @author Tuukka Lehtonen
 * 
 * @param <Factory> the factory type to use
 * 
 * @see ComparableFactoryResolver
 * @see LabelDecoratorFactoryResolver
 * @see ImageDecoratorFactoryResolver
 * @see LabelerFactoryResolver
 * @see ImagerFactoryResolver
 * @see ViewpointFactoryResolver
 */
public abstract class AbstractFactoryResolverQueryProcessor<Factory> extends AbstractNodeQueryProcessor<Collection<Factory>> {

    private final EvaluatorData data;
    private final QueryKey<Collection<Factory>> identifier;

    public AbstractFactoryResolverQueryProcessor(EvaluatorData data, QueryKey<Collection<Factory>> identifier) {
        this.data = data;
        this.identifier = identifier;
    }

    @Override
    public QueryKey<Collection<Factory>> getIdentifier() {
        return identifier;
    }

    @Override
    public Collection<Factory> query(final NodeQueryManager manager, final NodeContext context) {
        assert context != null;

        Object input = context.getConstant(BuiltinKeys.INPUT);
        assert input != null;

        Collection<Evaluator> evals = data.get(input);
        if (evals.isEmpty())
            return Collections.emptyList();

        ArrayList<Preference<Factory>> factories = new ArrayList<Preference<Factory>>(4);
        for (Evaluator eval : evals) {
            evaluateTree(manager, context, getEvaluatorTree(eval), factories);
        }

        if (factories.isEmpty())
            return Collections.emptyList();

        if (factories.size() > 1)
            Collections.sort(factories);

        ArrayList<Factory> result = new ArrayList<Factory>(factories.size());
        for (Preference<Factory> p : factories) {
            result.add(p.object);
        }

        return result;
    }

    protected void evaluateTree(NodeQueryManager manager, NodeContext context, EvaluatorTree<Factory> tree, Collection<Preference<Factory>> result) {
        Tester test = tree.getTester();

        if (test.test(manager, context)) {
            CollectionUtils.checkedAdd(tree.getAcceptedFactories(), result);

            Collection<EvaluatorTree<Factory>> children = tree.getChildren();
            if (children == null)
                return;
            for (EvaluatorTree<Factory> e : children) {
                evaluateTree(manager, context, e, result);
            }
        }
    }

    /**
     * @param evaluator
     * @return
     */
    protected abstract EvaluatorTree<Factory> getEvaluatorTree(Evaluator evaluator);

}
