package org.simantics.annotation.ui;

import java.io.File;
import java.io.IOException;
import java.util.Map;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.simantics.Simantics;
import org.simantics.annotation.ui.wizard.AnnotationTypeExporter;
import org.simantics.annotation.ui.wizard.AnnotationTypeImportWizard;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PropertyInfo;
import org.simantics.db.layer0.request.UnescapedPropertyMapOfResource;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.userComponent.ComponentTypeCommands;
import org.simantics.scl.runtime.tuple.Tuple2;
import org.simantics.utils.datastructures.Pair;

public class SCLAnnotation {
	

	public static Resource newAnnotationType(WriteGraph graph, Resource parent) throws DatabaseException {
		
		Pair<Resource, Resource> annotation = AnnotationUtils.newAnnotationType(graph, parent);
		Resource type = annotation.second;
		
		return type;
	}
	
	public static Tuple2 newAnnotationType1(WriteGraph graph, Resource parent) throws DatabaseException {
		
		Pair<Resource, Resource> annotation = AnnotationUtils.newAnnotationType(graph, parent);
		Tuple2 tuple = new Tuple2(annotation.first, annotation.second);
		
		return tuple;
	}
	
	public static Resource newAnnotationValue(WriteGraph graph, Resource parent, Resource property) throws DatabaseException {
		graph.markUndoPoint();
		Resource annotationValue = AnnotationUtils.newAnnotationInstance(graph, parent, property);
		return annotationValue;
	}

	public static Resource newAnnotationValueWithName(WriteGraph graph, Resource parent, String name, Resource property) throws DatabaseException {
		Resource existing = Layer0Utils.getPossibleChild(graph, parent, name);
		if(existing != null) return existing;
 		Resource annotationValue = AnnotationUtils.newAnnotationInstance(graph, parent, name, property);
		return annotationValue;
	}

	public static Resource newAnnotationProperty(WriteGraph graph, Resource annotationType) throws DatabaseException {
		
		Resource annotationProperty = ComponentTypeCommands.createPropertyWithDefaults(graph, annotationType);
		
		return annotationProperty;
	}
	
	public static Resource advancedAnnotationProperty(WriteGraph graph, Resource annotationType, String name, String value, String unit, String range, String label, String description) throws DatabaseException {
		
		Resource property = ComponentTypeCommands.createProperty(graph, annotationType, name, value, unit, range, label, description);
		
		return property;
	}
	
	public static void removeAnnotationProperty(WriteGraph graph, Resource property) throws DatabaseException {
		
		Resource annotationType = getAnnotationType(graph, property);
		ComponentTypeCommands.removeProperty(graph, annotationType, property);
	}
	
	public static void setPropertyType(WriteGraph graph, Resource property, String type) throws DatabaseException {
		
		Resource annotationType = getAnnotationType(graph, property);
		ComponentTypeCommands.setRequiredType(graph, property, type);
		ComponentTypeCommands.convertDefaultValue(graph, annotationType, property, type);
		
	}
	
	public static void setPropertyUnit(WriteGraph graph, Resource property, String unit) throws DatabaseException {
		
		Resource annotationType = getAnnotationType(graph, property);
		ComponentTypeCommands.setUnit(graph, property, annotationType, unit);
		
	}
	
	public static void setPropertyLabel(WriteGraph graph, Resource property, String label) throws DatabaseException {
		
		ComponentTypeCommands.setLabel(graph, property, label);
	}
	
	public static void setPropertyRange(WriteGraph graph, Resource property, String range) throws DatabaseException {
		
		Resource annotationType = getAnnotationType(graph, property);
		ComponentTypeCommands.setRange(graph, property, annotationType, range);
	}
	
	public static void setPropertyDescription(WriteGraph graph, Resource property, String description) throws DatabaseException {
		
		ComponentTypeCommands.setDescription(graph, property, description);
	}
	
	public static void setDefaultValue(WriteGraph graph, Resource property, String value) throws DatabaseException {
		
		Resource annotationType = getAnnotationType(graph, property);
		ComponentTypeCommands.setDefaultValue(graph, annotationType, property, value);
	}
	
	public static Resource attachAnnotation(WriteGraph graph, Resource parent, Resource annotationType) throws DatabaseException {

		Resource annotation = AnnotationUtils.newAnnotation(graph, parent, annotationType);
		
		return annotation;
	}
	
	public static Resource getAnnotationType(ReadGraph graph, Resource property) throws DatabaseException {
		
		Layer0 L0 = Layer0.getInstance(graph);
		Resource annotationType = graph.getPossibleObject(property, L0.PartOf);
		return annotationType;
		
	}
	
	public static Resource saveAnnotation(WriteGraph graph, Resource annotation, Resource library, String name) throws DatabaseException {
		return SCL.SaveModifier.saveAnnotation(graph, annotation, library, name);
	}
	
	public static void importAnnotationFromFile(File importLocation, Resource target) throws SerializationException, IOException, DatabaseException {
		Session session = Simantics.getSession();
		AnnotationTypeImportWizard.doImport(new NullProgressMonitor(), importLocation, session, target);
	}
	
	public static void exportAnnotationToFile(File location, Resource annotationType) throws DatabaseException, IOException {
		AnnotationTypeExporter.doExport(location, annotationType);
	}
	
	public static void rename(WriteGraph graph, Resource resource, String newName) throws DatabaseException {
		ComponentTypeCommands.rename(graph, resource, newName);
	}

    public static void copyAnnotationData(WriteGraph graph, Resource source, Resource target) throws DatabaseException {
        // source and target are expected to be of the same type
        Map<String, PropertyInfo> sps = graph.syncRequest(new UnescapedPropertyMapOfResource(source));
        Map<String, PropertyInfo> tps = graph.syncRequest(new UnescapedPropertyMapOfResource(target));

        for (PropertyInfo sp : sps.values()) {
            PropertyInfo tp = tps.get(sp.name);
            if (tp != null) {
                for (Statement srcstm : graph.getStatements(source, sp.predicate)) {
                    if (srcstm.isAsserted(source)) {
                        // Make sure the target value is asserted as well.
                        graph.deny(target, sp.predicate);
                    } else {
                        Resource property = srcstm.getObject();
                        // Remove existing property from source and link the target property
                        for (Statement tgtstm : graph.getStatements(target, tp.predicate)) {
                            if (!tgtstm.isAsserted(target))
                                graph.deny(tgtstm);
                        }
                        // Re-link the source property to the target entity.
                        graph.deny(source, sp.predicate, property);
                        graph.claim(target, tp.predicate, property);
                    }
                }
            }
        }
    }

}