/*******************************************************************************
 * This library is free software licensed under LGPL version 2.1
 * Based on GNU Trove, Copyright (c) 2001, Eric D. Friedman.
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package gnu.trove.ext;

import java.util.function.Predicate;


/**
 * An implementation of the <tt>Set</tt> interface that uses an
 * open-addressed hash table to store its contents.
 *
 * Created: Sat Nov  3 10:38:17 2001
 *
 * @author Eric D. Friedman
 * @version $Id: QueryIdentityHashSet.java,v 1.1 2008/03/14 11:32:01 tuoksk Exp $
 */

final public class IdentityHashSet<T> extends HashBase<T> {

	/**
	 * Creates a new <code>THashSet</code> instance with a prime
	 * capacity equal to or greater than <tt>initialCapacity</tt> and
	 * with the default load factor.
	 *
	 * @param initialCapacity an <code>int</code> value
	 */

	public IdentityHashSet(int initialCapacity, Class<T> clazz, T REMOVED) {
		super(initialCapacity, clazz, REMOVED);
	}

	/**
	 * Locates the index at which <tt>obj</tt> can be inserted.  if
	 * there is already a value equal()ing <tt>obj</tt> in the set,
	 * returns that value's index as <tt>-index - 1</tt>.
	 *
	 * @param obj an <code>Object</code> value
	 * @return the index of a FREE slot at which obj can be inserted
	 * or, if obj is already stored in the hash, the negative value of
	 * that index, minus 1: -index -1.
	 */
	final protected int insertionIndex(T obj) {

		final Object[] set = _set;
		final int length = set.length;
		final int hash = System.identityHashCode(obj) & 0x7fffffff;
		int index = hash % length;
		Object cur = set[index];

		if (cur == null) {
			return index;       // empty, all done
		} else if (cur != REMOVED && (cur == obj)) {
			return -index -1;   // already stored
		} else {                // already FULL or REMOVED, must probe
			// compute the double hash
			final int probe = 1 + (hash % (length - 2));

			// if the slot we landed on is FULL (but not removed), probe
			// until we find an empty slot, a REMOVED slot, or an element
			// equal to the one we are trying to insert.
			// finding an empty slot means that the value is not present
			// and that we should use that slot as the insertion point;
			// finding a REMOVED slot means that we need to keep searching,
			// however we want to remember the offset of that REMOVED slot
			// so we can reuse it in case a "new" insertion (i.e. not an update)
			// is possible.
			// finding a matching value means that we've found that our desired
			// key is already in the table
			if (cur != REMOVED) {
				// starting at the natural offset, probe until we find an
				// offset that isn't full.
				do {
					index -= probe;
					if (index < 0) {
						index += length;
					}
					cur = set[index];
				} while (cur != null
						&& cur != REMOVED
						&& (cur != obj));
			}

			// if the index we found was removed: continue probing until we
			// locate a free location or an element which equal()s the
			// one we have.
			if (cur == REMOVED) {
				int firstRemoved = index;
				while (cur != null
						&& (cur == REMOVED || (cur != obj))) {
					index -= probe;
					if (index < 0) {
						index += length;
					}
					cur = set[index];
				}
				// NOTE: cur cannot == REMOVED in this block
				return (cur != null) ? -index -1 : firstRemoved;
			}
			// if it's full, the key is already stored
			// NOTE: cur cannot equal REMOVE here (would have retuned already (see above)
			return (cur != null) ? -index -1 : index;
		}
	}

	/**
	 * Inserts a value into the set.
	 *
	 * @param obj an <code>Object</code> value
	 * @return true if the set was modified by the add operation
	 */
	final public boolean add(final T obj) {

		final int index = insertionIndex(obj);

		if (index < 0) {
			return false;       // already present in set, nothing to add
		}

		final T old = _set[index];
		_set[index] = obj;

		postInsertHook(old == null);
		return true;            // yes, we added something

	}

	/**
	 * Expands the set to accomodate new values.
	 *
	 * @param newCapacity an <code>int</code> value
	 */
	final protected void rehash(int newCapacity) {
		final int oldCapacity = _set.length;
		final T oldSet[] = _set;
		_set = createArray(newCapacity);
		for (int i = oldCapacity; i-- > 0;) {
			if(oldSet[i] != null && oldSet[i] != REMOVED) {
				final T o = oldSet[i];
				int index = insertionIndex(o);
				if (index < 0) {
					throwObjectContractViolation(_set[(-index -1)], o);
				}
				_set[index] = o;
			}
		}
	}

	/**
	 * Locates the index of <tt>obj</tt>.
	 *
	 * @param obj an <code>Object</code> value
	 * @return the index of <tt>obj</tt> or -1 if it isn't in the set.
	 */
	final protected int index(Object obj) {

		final Object[] set = _set;
		final int length = set.length;
		final int hash = System.identityHashCode(obj) & 0x7fffffff;
		int index = hash % length;
		Object cur = set[index];

		if ( cur == null ) return -1;

		// NOTE: here it has to be REMOVED or FULL (some user-given value)
		if ( cur == REMOVED || (cur != obj)) {
			// see Knuth, p. 529
			final int probe = 1 + (hash % (length - 2));

			do {
				index -= probe;
				if (index < 0) {
					index += length;
				}
				cur = set[index];
			} while (cur != null
					&& (cur == REMOVED || (cur != obj)));
		}

		return cur == null ? -1 : index;
	}

	/**
	 * Removes <tt>obj</tt> from the set.
	 *
	 * @param obj an <code>Object</code> value
	 * @return true if the set was modified by the remove operation.
	 */
	final public boolean remove(Object obj) {
		int index = index(obj);
		if (index >= 0) {
			removeAt(index);
			return true;
		}
		return false;
	}

	final public T removeOrGetByPredicate(Predicate<T> predicate) {
		tempDisableAutoCompaction();
		try {
			for(int i=0;i<_set.length;i++) {
				T entry = _set[i];
				if(entry != null && REMOVED != entry) {
					if(!predicate.test(entry))
						return entry; 
					removeAt(i);
				}
			}
			return null;
		} finally {
			reenableAutoCompaction(false);
		}
	}

	final public T first() {
		for(int i=0;i<_set.length;i++) {
			T entry = _set[i];
			if(entry != null && REMOVED != entry) {
				return entry; 
			}
		}
		return null;
	}

	final public T firstBy(Predicate<T> predicate) {
		for(int i=0;i<_set.length;i++) {
			T entry = _set[i];
			if(entry != null && REMOVED != entry) {
				if(predicate.test(entry))
					return entry; 
			}
		}
		return null;
	}

	final public int removeBy(Predicate<T> predicate) {
		tempDisableAutoCompaction();
		int size = 0;
		try {
			for(int i=0;i<_set.length;i++) {
				T entry = _set[i];
				if(entry != null && REMOVED != entry) {
					if(predicate.test(entry))
						removeAt(i);
					else size++;
				}
			}
		} finally {
			reenableAutoCompaction(false);
		}
		return size;
	}

}
