/*******************************************************************************
 * This library is free software licensed under LGPL version 2.1
 * Based on GNU Trove, Copyright (c) 2001, Eric D. Friedman.
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package gnu.trove.ext;

import java.lang.reflect.Array;
import java.util.function.Consumer;

import gnu.trove.impl.HashFunctions;
import gnu.trove.impl.hash.THash;

public abstract class HashBase<T> extends THash {

	protected transient T[] _set;

	protected final T REMOVED;

	private final Class<T> clazz;

	public HashBase(Class<T> clazz, T REMOVED) {
		super(DEFAULT_CAPACITY, 0.75f);
		this.clazz = clazz;
		this.REMOVED = REMOVED;
		setUp( HashFunctions.fastCeil( DEFAULT_CAPACITY / 0.75f ) );
	}

	public HashBase(int initialCapacity, Class<T> clazz, T REMOVED) {
		super(initialCapacity, 0.75f);
		this.clazz = clazz;
		this.REMOVED = REMOVED;
		setUp( HashFunctions.fastCeil( initialCapacity / 0.75f ) );
	}

	@Override
	public final int capacity() {
		return _set.length;
	}

	@Override
	protected final void removeAt(int index) {
		_set[index] = REMOVED;
		super.removeAt(index);
	}

	/**
	 * initializes the Object set of this hash table.
	 *
	 * @param initialCapacity an <code>int</code> value
	 * @return an <code>int</code> value
	 */
	@Override
	protected final int setUp(int initialCapacity) {
		if(clazz == null)
			return initialCapacity;
		int capacity;
		capacity = super.setUp(initialCapacity);
		if(clazz != null)
			_set = createArray(capacity);
		return capacity;
	}

	@SuppressWarnings("unchecked")
	protected final T[] createArray(int capacity) {
		return (T[])Array.newInstance(clazz, capacity);
	}

	/**
	 * Convenience methods for subclasses to use in throwing exceptions about
	 * badly behaved user objects employed as keys.  We have to throw an
	 * IllegalArgumentException with a rather verbose message telling the
	 * user that they need to fix their object implementation to conform
	 * to the general contract for java.lang.Object.
	 *
	 * @param o1 the first of the equal elements with unequal hash codes.
	 * @param o2 the second of the equal elements with unequal hash codes.
	 * @exception IllegalArgumentException the whole point of this method.
	 */
	protected final void throwObjectContractViolation(Object o1, Object o2)
			throws IllegalArgumentException {
		throw new IllegalArgumentException("Equal objects must have equal hashcodes. "
				+ "During rehashing, Trove discovered that "
				+ "the following two objects claim to be "
				+ "equal (as in java.lang.Object.equals()) "
				+ "but their hashCodes (or those calculated by "
				+ "your TObjectHashingStrategy) are not equal."
				+ "This violates the general contract of "
				+ "java.lang.Object.hashCode().  See bullet point two "
				+ "in that method's documentation. "
				+ "object #1 =" + o1 + " object #1 hash = " + o1.hashCode() + " object #1 id = " + System.identityHashCode(o1)
				+ "; object #2 =" + o2 + " object #2 hash = " + o2.hashCode() + " object #2 id = " + System.identityHashCode(o2));
	}

	public final void values(Consumer<? super T> result) {
		for (int i = _set.length; i-- > 0;) {
			T entry = _set[i];
			if(entry != null && entry != REMOVED) {
				result.accept(entry);
			}
		}
	}

}
