/*******************************************************************************
 * This library is free software licensed under LGPL version 2.1
 * Based on GNU Trove, Copyright (c) 2001, Eric D. Friedman.
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - improvements
 *******************************************************************************/
package gnu.trove.ext;

import java.lang.reflect.Array;
import java.util.function.Consumer;

import gnu.trove.impl.HashFunctions;
import gnu.trove.impl.hash.THash;

public final class DualIdContainerHashMapWithPrimaryIdLookup<T extends DualIdContainer> extends THash {

	protected transient DualIdContainerHashMapWithPrimaryId<T>[] _set;

	protected final DualIdContainerHashMapWithPrimaryId<T> REMOVED;

	protected final Class<T> clazz;

	private int sizeInternal = 0;

	/**
	 * Creates a new <code>THashMap</code> instance with the default
	 * capacity and load factor.
	 */
	public DualIdContainerHashMapWithPrimaryIdLookup(Class<T> clazz) throws Exception {
		super(DEFAULT_CAPACITY, 0.75f);
		this.clazz = clazz;
		this.REMOVED = new DualIdContainerHashMapWithPrimaryId<T>(-1, clazz);
		setUp( HashFunctions.fastCeil( DEFAULT_CAPACITY / 0.75f ) );
	}

	@Override
	public final int capacity() {
		return _set.length;
	}

	@Override
	protected final void removeAt(int index) {
		_set[index] = REMOVED;
		super.removeAt(index);
	}

	/**
	 * initializes the Object set of this hash table.
	 *
	 * @param initialCapacity an <code>int</code> value
	 * @return an <code>int</code> value
	 */
	@Override
	protected final int setUp(int initialCapacity) {
		if(clazz == null)
			return initialCapacity;
		int capacity;
		capacity = super.setUp(initialCapacity);
		if(clazz != null)
			_set = createArray(capacity);
		return capacity;
	}

	@SuppressWarnings("unchecked")
	protected final DualIdContainerHashMapWithPrimaryId<T>[] createArray(int capacity) {
		return new DualIdContainerHashMapWithPrimaryId[capacity];
	}

	/**
	 * Convenience methods for subclasses to use in throwing exceptions about
	 * badly behaved user objects employed as keys.  We have to throw an
	 * IllegalArgumentException with a rather verbose message telling the
	 * user that they need to fix their object implementation to conform
	 * to the general contract for java.lang.Object.
	 *
	 * @param o1 the first of the equal elements with unequal hash codes.
	 * @param o2 the second of the equal elements with unequal hash codes.
	 * @exception IllegalArgumentException the whole point of this method.
	 */
	protected final void throwObjectContractViolation(Object o1, Object o2)
			throws IllegalArgumentException {
		throw new IllegalArgumentException("Equal objects must have equal hashcodes. "
				+ "During rehashing, Trove discovered that "
				+ "the following two objects claim to be "
				+ "equal (as in java.lang.Object.equals()) "
				+ "but their hashCodes (or those calculated by "
				+ "your TObjectHashingStrategy) are not equal."
				+ "This violates the general contract of "
				+ "java.lang.Object.hashCode().  See bullet point two "
				+ "in that method's documentation. "
				+ "object #1 =" + o1 + " object #1 hash = " + o1.hashCode() + " object #1 id = " + System.identityHashCode(o1)
				+ "; object #2 =" + o2 + " object #2 hash = " + o2.hashCode() + " object #2 id = " + System.identityHashCode(o2));
	}

	private final DualIdContainerHashMapWithPrimaryId<T> putMap(final int id, final DualIdContainerHashMapWithPrimaryId<T> value) {
		DualIdContainerHashMapWithPrimaryId<T> previous = null;
		Object oldKey;
		int index = insertionIndex(id);
		boolean isNewMapping = true;
		if (index < 0) {
			index = -index -1;
			previous = _set[index];
			isNewMapping = false;
		}
		oldKey = _set[index];
		_set[index] = value;
		if (isNewMapping) {
			postInsertHook(oldKey == null);
		}
		return previous;
	}

	private final int removalIndex(final int id) {

		final DualIdContainerHashMapWithPrimaryId<T>[] set = _set;
		final int length = set.length;
		final int hash = (31 * id) & 0x7fffffff;
		int index = hash % length;
		DualIdContainerHashMapWithPrimaryId<T> cur = set[index];

		if ( cur == null ) return -1;

		// NOTE: here it has to be REMOVED or FULL (some user-given value)
		if ( cur == REMOVED || !(id == cur.primaryId)) {
			// see Knuth, p. 529
			final int probe = 1 + (hash % (length - 2));

			do {
				index -= probe;
				if (index < 0) {
					index += length;
				}
				cur = set[index];
			} while (cur != null
					&& (cur == REMOVED || !(id == cur.primaryId)));
		}

		return cur == null ? -1 : index;

	}


	/**
	 * Deletes a key/value pair from the map.
	 *
	 * @param key an <code>Object</code> value
	 * @return an <code>Object</code> value
	 */
	private final Object removeMap(final int id) {
		DualIdContainerHashMapWithPrimaryId<T> prev = null;
		int index = removalIndex(id);
		if (index >= 0) {
			prev = _set[index];
			removeAt(index);    // clear key,state; adjust size
			sizeInternal-=prev.size();
		}
		return prev;
	}

	/**
	 * Locates the index at which <tt>obj</tt> can be inserted.  if
	 * there is already a value equal()ing <tt>obj</tt> in the set,
	 * returns that value's index as <tt>-index - 1</tt>.
	 *
	 * @param obj an <code>Object</code> value
	 * @return the index of a FREE slot at which obj can be inserted
	 * or, if obj is already stored in the hash, the negative value of
	 * that index, minus 1: -index -1.
	 */
	private int insertionIndex(final int id) {

		final DualIdContainerHashMapWithPrimaryId<T>[] set = _set;
		final int length = set.length;
		final int hash = (31 * id) & 0x7fffffff;
		int index = hash % length;
		DualIdContainerHashMapWithPrimaryId<T> cur = set[index];

		if (cur == null) {
			return index;       // empty, all done
		} else if (cur != REMOVED && (id == cur.primaryId)) {
			return -index -1;   // already stored
		} else {                // already FULL or REMOVED, must probe
			// compute the double hash
			final int probe = 1 + (hash % (length - 2));

			// if the slot we landed on is FULL (but not removed), probe
			// until we find an empty slot, a REMOVED slot, or an element
			// equal to the one we are trying to insert.
			// finding an empty slot means that the value is not present
			// and that we should use that slot as the insertion point;
			// finding a REMOVED slot means that we need to keep searching,
			// however we want to remember the offset of that REMOVED slot
			// so we can reuse it in case a "new" insertion (i.e. not an update)
			// is possible.
			// finding a matching value means that we've found that our desired
			// key is already in the table
			if (cur != REMOVED) {
				// starting at the natural offset, probe until we find an
				// offset that isn't full.
				do {
					index -= probe;
					if (index < 0) {
						index += length;
					}
					cur = set[index];
				} while (cur != null
						&& cur != REMOVED
						&& ! (id == cur.primaryId));
			}

			// if the index we found was removed: continue probing until we
			// locate a free location or an element which equal()s the
			// one we have.
			if (cur == REMOVED) {
				int firstRemoved = index;
				while (cur != null
						&& (cur == REMOVED || ! (id == cur.primaryId))) {
					index -= probe;
					if (index < 0) {
						index += length;
					}
					cur = set[index];
				}
				// NOTE: cur cannot == REMOVED in this block
				return (cur != null) ? -index -1 : firstRemoved;
			}
			// if it's full, the key is already stored
			// NOTE: cur cannot equal REMOVE here (would have retuned already (see above)
			return (cur != null) ? -index -1 : index;
		}
	}

	private final DualIdContainerHashMapWithPrimaryId<T> getMap(final int primaryId) {

		final DualIdContainerHashMapWithPrimaryId<T>[] set = _set;
		final int length = set.length;
		final int hash = (31 * primaryId) & 0x7fffffff;
		int index = hash % length;
		DualIdContainerHashMapWithPrimaryId<T> cur = set[index];

		if ( cur == null ) return null;

		// NOTE: here it has to be REMOVED or FULL (some user-given value)
		if ( cur == REMOVED || (primaryId != cur.primaryId)) {
			// see Knuth, p. 529
			final int probe = 1 + (hash % (length - 2));

			do {
				index -= probe;
				if (index < 0) {
					index += length;
				}
				cur = set[index];
			} while (cur != null
					&& (cur == REMOVED || (primaryId != cur.primaryId)));
		}

		return cur;

	}

	private final int rehashInsertionIndex(final int id, final DualIdContainerHashMapWithPrimaryId<T>[] set) {

		final int length = set.length;
		final int hash = (31 * id) & 0x7fffffff;
		int index = hash % length;
		DualIdContainerHashMapWithPrimaryId<T> cur = set[index];

		if (cur == null) {
			return index;       // empty, all done
		} else if (cur != REMOVED && (id == cur.primaryId)) {
			return -index -1;   // already stored
		} else {                // already FULL or REMOVED, must probe
			// compute the double hash
			final int probe = 1 + (hash % (length - 2));

			// if the slot we landed on is FULL (but not removed), probe
			// until we find an empty slot, a REMOVED slot, or an element
			// equal to the one we are trying to insert.
			// finding an empty slot means that the value is not present
			// and that we should use that slot as the insertion point;
			// finding a REMOVED slot means that we need to keep searching,
			// however we want to remember the offset of that REMOVED slot
			// so we can reuse it in case a "new" insertion (i.e. not an update)
			// is possible.
			// finding a matching value means that we've found that our desired
			// key is already in the table
			if (cur != REMOVED) {
				// starting at the natural offset, probe until we find an
				// offset that isn't full.
				do {
					index -= probe;
					if (index < 0) {
						index += length;
					}
					cur = set[index];
				} while (cur != null
						&& cur != REMOVED
						&& ! (id == cur.primaryId));
			}

			// if the index we found was removed: continue probing until we
			// locate a free location or an element which equal()s the
			// one we have.
			if (cur == REMOVED) {
				int firstRemoved = index;
				while (cur != null
						&& (cur == REMOVED || ! (id == cur.primaryId))) {
					index -= probe;
					if (index < 0) {
						index += length;
					}
					cur = set[index];
				}
				// NOTE: cur cannot == REMOVED in this block
				return (cur != null) ? -index -1 : firstRemoved;
			}
			// if it's full, the key is already stored
			// NOTE: cur cannot equal REMOVE here (would have retuned already (see above)
			return (cur != null) ? -index -1 : index;
		}
	}

	/**
	 * rehashes the map to the new capacity.
	 *
	 * @param newCapacity an <code>int</code> value
	 */
	@SuppressWarnings("unchecked")
	protected void rehash(int newCapacity) {

		int oldCapacity = _set.length;
		DualIdContainerHashMapWithPrimaryId<T> oldSet[] = _set;
		DualIdContainerHashMapWithPrimaryId<T> newSet[]  = (DualIdContainerHashMapWithPrimaryId[])Array.newInstance(DualIdContainerHashMapWithPrimaryId.class, newCapacity);

		for (int i = oldCapacity; i-- > 0;) {
			if(oldSet[i] != null && oldSet[i] != REMOVED) {
				DualIdContainerHashMapWithPrimaryId<T> o = oldSet[i];
				int index = rehashInsertionIndex(o.primaryId, newSet);
				if (index < 0) {
					throwObjectContractViolation(newSet[(-index -1)], o);
				}
				newSet[index] = o;
			}
		}

		_set = newSet;

	}

	/**
	 * Inserts a key/value pair into the map.
	 *
	 * @param key an <code>Object</code> value
	 * @param value an <code>Object</code> value
	 * @return the previous value associated with <tt>key</tt>,
	 * or null if none was found.
	 */
	public T put(final long id, T value) {
		int primaryId = primaryId(id);
		DualIdContainerHashMapWithPrimaryId<T> map = getMap(primaryId);
		if(map == null) {
			try {
				map = new DualIdContainerHashMapWithPrimaryId<T>(primaryId, clazz);
				putMap(primaryId, map);
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
		}
		T old = map.put(id, value);
		if(old == null) sizeInternal++;
		return old;
	}

	final protected static long longId(long primaryId, long secondaryId) {
		long result = (primaryId<<32) | (secondaryId & 0xffffffffL); 
		return result;
	}

	final public int primaryId(final long id) {
		return (int)(id>>>32);
	}

	public final T get(final int r1, final int r2) {
		DualIdContainerHashMapWithPrimaryId<T> map = getMap(r1);
		if(map == null) return null;
		return map.get(longId(r1,r2));
	}

	public final T remove(final long id) {
		int r1 = primaryId(id);
		DualIdContainerHashMapWithPrimaryId<T> map = getMap(r1);
		if(map == null)
			return null;
		T removed = map.remove(id);
		if(removed != null)
			sizeInternal--;
		if(map.isEmpty())
			removeMap(r1);
		return removed;
	}

	@Override
	public final int size() {
		return sizeInternal;
	}

	public final void values(Consumer<? super T> result) {

		for (int i = _set.length; i-- > 0;) {
			if(_set[i] != null && _set[i] != REMOVED) {
				DualIdContainerHashMapWithPrimaryId<T> map = _set[i];
				map.values(result);
			}
		}

	}

	public final void values(int primaryId, Consumer<? super T> result) {

		DualIdContainerHashMapWithPrimaryId<T> map = getMap(primaryId);
		if(map != null)
			map.values(result);

	}

}
