/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.structural.rules.domain;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.objmap.exceptions.MappingException;
import org.simantics.objmap.graph.rules.domain.IDomainAccessor;
import org.simantics.objmap.graph.rules.domain.MappingUtils;
import org.simantics.objmap.structural.StructuralResource;


public class StructuralRelatedObjectsAccessor implements IDomainAccessor<StructuralResource,Collection<StructuralResource>> {

    static Logger LOGGER = LoggerFactory.getLogger(StructuralRelatedObjectsAccessor.class);
    
	Resource relation;
	boolean deleteExtraObjects;

	public StructuralRelatedObjectsAccessor(Resource relation, boolean deleteExtraObjects) {
        super();
        this.relation = relation;
        this.deleteExtraObjects = deleteExtraObjects;
    }

    @Override
	public Collection<StructuralResource> get(ReadGraph g, StructuralResource element) throws MappingException {
		try {
		    LOGGER.trace("        RelatedObjectsAccessor.get");
			
		    if (!element.isStructural())
		    	return Collections.emptyList();
		    
		    // Structural instance
		    Resource instance = StructuralUtils.getContainingInstance(element);
		    
		    Resource publicRelation = StructuralUtils.getPublishedRelation(g, element, relation);
		    
		    if (publicRelation == null)
		      return Collections.emptyList();
		    
		    Collection<Resource> coll = g.getObjects(instance, publicRelation);
			List<StructuralResource> result = new ArrayList<StructuralResource>(coll.size());
			List<Resource> context = new ArrayList<Resource>();
			for (int i = 0; i < element.getContext().size()-1; i++)
				context.add(element.getContext().get(i));
			for (Resource r : coll) {
				if (StructuralUtils.isStructuralInstance(g, r)) {
					result.add(new StructuralResource(g, r, context,r));
				} else {
					result.add(new StructuralResource(g, r, context));
				}
			}
			return result;
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
	}
	
	@Override
	public boolean set(WriteGraph g, StructuralResource element, Collection<StructuralResource> value)
			throws MappingException {
		try {
		    LOGGER.trace("        RelatedObjectsAccessor.set");
		    
		    if (!element.isStructural())
		    	return false;
		    
		    Resource instance = StructuralUtils.getContainingInstance(element);
		    Resource publicRelation = null;
		    if (value.size() == 0) {
		    	publicRelation = StructuralUtils.getPublishedRelation(g, element, relation);
		    	if (publicRelation == null)
		    		return false;
		    	else {
		    		return MappingUtils.synchronizeStatements(g, instance, publicRelation, new Resource[0], deleteExtraObjects);
		    	}
		    } else {
		    	publicRelation = StructuralUtils.getOrCreatePublishedRelation(g, element, relation);
		    	if (publicRelation == null)
		    		throw new MappingException("Structural Resource " + element + " cannot contain structural elements, the Resource is not published.");
			    Resource[] arr = new Resource[value.size()];
			    int i = 0; 
			    for (StructuralResource sr : value) {
			    	arr[i++] = sr.getResource();
			    }
			    return MappingUtils.synchronizeStatements(g, instance, publicRelation, arr, deleteExtraObjects);
		    }
		    
		    
			
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
		
	}

	
}
