/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.graph.rules.domain;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.simantics.databoard.binding.Binding;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ServiceException;
import org.simantics.layer0.Layer0;
import org.simantics.objmap.exceptions.MappingException;
import org.simantics.objmap.graph.annotations.factories.DataTypeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * @author Marko Luukkainen
 */
public class CompoundValueAccessor implements IDomainAccessor<Resource,Object> {

    static final Logger LOGGER = LoggerFactory.getLogger(CompoundValueAccessor.class);
    
	Resource objRelation;
	Resource objType;
	Resource valRelation;
	Binding valueBinding;

	public CompoundValueAccessor(Resource objRelation, Resource objType, Resource valRelation, Binding valueBinding) {
		this.objRelation = objRelation;
		this.objType = objType;
		this.valRelation = valRelation;
		this.valueBinding = valueBinding;
	}

	@Override
	public Object get(ReadGraph g, Resource element) throws MappingException {
		try {
			Layer0 l0 = Layer0.getInstance(g);
		    LOGGER.trace("        CompoundValueAccessor.get");
		    Collection<Statement> coll = g.getStatements(element, objRelation);
		    Map<String,Object> map = new HashMap<String, Object>();
		    for (Statement c : coll) {
		    	String name = g.getRelatedValue(c.getObject(), l0.HasName);
		    	if (!map.containsKey(name) || !c.isAsserted(element)) {
					final Object value = getValue(g, c.getObject());
					map.put(name, value);
				}
		    }
			return map;
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
	}

	@Override
	public boolean set(WriteGraph g, Resource element, Object v)
			throws MappingException {
		try {
			Layer0 l0 = Layer0.getInstance(g);
		    LOGGER.trace("        CompoundValueAccessor.set");
		    @SuppressWarnings("unchecked")
		    Map<String,Object> values = (Map<String, Object>)v;
		    
		    Collection<Statement> coll = g.getStatements(element, objRelation);
		    Map<String,Statement> stmMap = new HashMap<String, Statement>();
		    Map<String,Object> valueMap = new HashMap<String, Object>();
		    for (Statement c : coll) {
		    	String name = g.getRelatedValue(c.getObject(), l0.HasName);
		    	if (!stmMap.containsKey(name) || !c.isAsserted(element)) {
		    		stmMap.put(name, c);
		    		valueMap.put(name, g.getRelatedValue(c.getObject(), valRelation));
		    	}
		    }
		    
		    boolean changed = false;
		    for (String key : values.keySet()) {
		    	Object value = values.get(key);
		    	if (value.equals(valueMap.get(key)))
		    		continue;
		    	changed = true;
		    	Statement stm = stmMap.get(key);
		    	if (stm == null || stm.isAsserted(element)) {
		    		Resource obj = g.newResource();
		    		g.claim(obj, l0.InstanceOf, objType);
		    		g.claimLiteral(obj, l0.HasName, key);
		    		g.claim(element, objRelation, obj);
		    		stm = getStatement(g, element, objRelation, obj);
		    	}
		    	
		    	Statement valueStatement = g.getPossibleStatement(stm.getObject(), valRelation);
		    	Resource valueType = valueBinding != null ?
		    			DataTypeUtils.dataTypeOfDatatype(g, valueBinding.type()) :
		    			DataTypeUtils.dataTypeOfClass(g, value.getClass());
		    			
				if(valueStatement == null) {
					Resource valueResource = g.newResource();
					g.claim(valueResource, Layer0.getInstance(g).InstanceOf, null, valueType);
					g.claim(stm.getObject(), valRelation, valueResource);
					claimValue(g, valueResource, value);
				} else {
					if (!valueStatement.isAsserted(stm.getObject()))
						g.claimValue(valueStatement.getObject(), value);
					else {
						Resource valueResource = g.newResource();
						g.claim(valueResource, Layer0.getInstance(g).InstanceOf, null, valueType);
						g.claim(stm.getObject(), valRelation, valueResource);
						claimValue(g, valueResource, value);
					}
				}
		    }
		    
		    return changed;
		} catch (DatabaseException e) {
			throw new MappingException(e);
		}
		
	}

	private void claimValue(WriteGraph g, Resource valueResource, Object value) throws ServiceException {
		if (valueBinding != null)
			g.claimValue(valueResource, value, valueBinding);
		else
			g.claimValue(valueResource, value);
	}
	
	private Statement getStatement(ReadGraph g, Resource s, Resource p, Resource o) throws DatabaseException{
		for (Statement stm : g.getStatements(s, p)) {
			if (stm.getObject().equals(o))
				return stm;
		}
		return null;
	}
	
	private Object getValue(ReadGraph g, final Resource object) throws DatabaseException {
		return valueBinding != null ? g.getRelatedValue(object, valRelation, valueBinding) : g.getRelatedValue(object, valRelation);
	}

}
