/*******************************************************************************
 * Copyright (c) 2007, 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.objmap.graph;

import java.util.Collection;
import java.util.Set;

import org.simantics.db.Disposable;
import org.simantics.db.ReadGraph;
import org.simantics.db.WriteGraph;
import org.simantics.objmap.bidirectional.IBidirectionalMapping;
import org.simantics.objmap.exceptions.MappingException;

/**
 * A mapping consists of domain (a set of resources), range (a set of Java objects) and
 * a set of links relating them. The mapping is used to propagate modifications of
 * domain elements to range and vice versa. 
 * 
 * @see <a href="http://www.simantics.org/wiki/index.php/org.simantics.objmap_Manual#Concepts">Manual</a>
 * 
 * @author Hannu Niemist
 */
public interface IMapping<Domain,Range> extends Disposable, IBidirectionalMapping<Domain, Range> {

    /**
     * Returns the domain of the mapping. All set operations are supported.
     * Adding a new domain element does not automatically create a link to it.
     * Removal of a domain element removes also a link and the target element,
     * but does not remove the element from the database.
     */
    Set<Domain> getDomain();

    /**
     * Returns the range of the mapping. All set operations are supported.
     * Adding a new range element does not automatically create a link to it.
     * Removal of a range element removes also a link and the domain element,
     * but does not remove the domain element from the database.
     */
    Set<Range> getRange();

    /**
     * Updates all domain elements whose counterpart is modified and creates new
     * domain elements for previously added range elements. Returns the
     * collection of domain elements that were modified or created in the update
     * process.
     */
    Collection<Domain> updateDomain(WriteGraph g) throws MappingException;

    /**
     * Updates all range elements whose counterpart is modified and creates new
     * range elements for previously added domain elements. Returns the
     * collection of range elements that were modified or created in the update
     * process.
     */
    Collection<Range> updateRange(ReadGraph g) throws MappingException;

    /**
     * Returns the counterpart of a domain element or null if the element does
     * not belong to the domain or does not have a link.
     */
    Range get(Domain domainElement);

    /**
     * Returns the counterpart of a range element or null if the element does
     * not belong to the range or does not have a link.
     */
    Domain inverseGet(Range rangeElement);

    /**
     * A convenience method that adds a domain element to the mapping and
     * immediately updates the mapping and returns the corresponding range
     * element.
     */
    Range map(ReadGraph g, Domain domainElement) throws MappingException;

    /**
     * A convenience method that adds a range element to the mapping and
     * immediately updates the mapping and returns the corresponding domain
     * element.
     */
    Domain inverseMap(WriteGraph g, Range rangeElement)
            throws MappingException;

    /**
     * Tells the mapping that the domain element has been modified.
     */
    void domainModified(Domain domainElement);

    /**
     * Tells the mapping that the range element has been modified.
     */
    void rangeModified(Range rangeElement);

    /**
     * Tells if some domain elements have been modified or added.
     */
    boolean isDomainModified();

    /**
     * Tells if some range elements have been modified or added.
     */
    boolean isRangeModified();
    
    Collection<Domain> getDomainModified();
    Collection<Range> getRangeModified();

    /**
     * Returns a collection of domain elements which have been modified and also
     * their counterparts in the mapping are modified. These elements are in
     * conflict in the sense that the updating domain and range in different
     * orders may produce different results.
     */
    Collection<Domain> getConflictingDomainElements();

    /**
     * Returns a collection of range elements which have been modified and also
     * their counterparts in the mapping are modified. These elements are in
     * conflict in the sense that the updating domain and range in different
     * orders may produce different results.
     */
    Collection<Range> getConflictingRangeElements();

    /**
     * Adds a listener for domain and range modifications.
     */
    void addMappingListener(IMappingListener listener);

    /**
     * Removes a previously added listener.
     */
    void removeMappingListener(IMappingListener listener);

}
