/*******************************************************************************
 * Copyright (c) 2014 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *     Semantum Oy - enabled template unassignment (#5030)
 *     Semantum Oy - multi-selection support (#5425)
 *******************************************************************************/
package org.simantics.modeling.template2d.ui.actions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.viewers.BaseLabelProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;
import org.simantics.NameLabelUtil;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.ActionFactory;
import org.simantics.db.layer0.adapter.ActionFactory2;
import org.simantics.db.layer0.adapter.Instances;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.template2d.ontology.Template2dResource;
import org.simantics.modeling.template2d.ui.Activator;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.ui.workbench.dialogs.ResourceListDialog;
import org.simantics.utils.datastructures.Pair;

import gnu.trove.set.hash.THashSet;

public class AssignDrawingTemplate implements ActionFactory, ActionFactory2 {

    @Override
    public Runnable create(Collection<?> targets) {
        final List<Resource> resources = new ArrayList<Resource>();
        for (Object target : targets) {
            if (!(target instanceof Resource))
                return null;
            resources.add((Resource) target);
        }
        return (targets.isEmpty()) ? null : assign(resources);
    }

    @Override
    public Runnable create(Object target) {
        if(!(target instanceof Resource))
            return null;
        return assign(Collections.singletonList((Resource) target));
    }

    private Runnable assign(final Collection<Resource> composites) {
        return new Runnable() {
            @Override
            public void run() {
                Simantics.getSession().asyncRequest(new ReadRequest() {
                    @Override
                    public void run(ReadGraph graph) throws DatabaseException {
                        assign(graph, composites);
                    }
                });
            }
        };
    }

    private void assign(ReadGraph graph, final Collection<Resource> composites) throws DatabaseException {
        Template2dResource TEMPLATE2D = Template2dResource.getInstance(graph);
        StructuralResource2 STR = StructuralResource2.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);

        Instances query = graph.adapt(TEMPLATE2D.DrawingTemplate, Instances.class);
        Resource root = null;
        Set<Resource> initialResources = new THashSet<Resource>();

        for (Resource composite : composites) {
            if (!graph.isInstanceOf(composite, STR.Composite))
                return;

            // Ensure all composites are part of the same index root.
            Resource r = graph.sync(new PossibleIndexRoot(composite));
            if (r == null || (root != null && !r.equals(root)))
                return;
            root = r;

            // Ensure each composite has a diagram.
            Resource diagram = graph.getPossibleObject(composite, MOD.CompositeToDiagram);
            if (diagram == null)
                return;

            Resource initial = graph.getPossibleObject(diagram, TEMPLATE2D.HasDrawingTemplate);
            if (initial != null)
                initialResources.add(initial);
        }

        final List<Pair<Resource, String>> elements = new ArrayList<Pair<Resource, String>>();

        for (Resource template : query.find(graph, root)) {
            String name = NameLabelUtil.modalName(graph, template);
            elements.add(Pair.make(template, name));
        }

        Resource init = initialResources.size() == 1 ? initialResources.iterator().next() : null;
        final Pair<Resource, String> initialSelection = init != null
                ? Pair.make(init, NameLabelUtil.modalName(graph, init))
                : null;

        Display.getDefault().asyncExec(new Runnable() {
            @Override
            public void run() {
                queryDrawingTemplate(composites, elements, initialSelection);
            }
        });
    }

    private static void queryDrawingTemplate(
            final Collection<Resource> composites,
            List<Pair<Resource, String>> elements,
            Pair<Resource, String> initialSelection
            )
    {
        Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
        ResourceListDialog dialog = new ResourceListDialog(shell, "Select template from list", "Select diagram template to assign " + ResourceListDialog.DEFAULT_LABEL_SUFFIX, new LabelProvider()) {
            @Override
            protected void createButtonsForButtonBar(Composite parent) {
                Button unassign = createButton(parent, IDialogConstants.CLIENT_ID, "Unassign", false);
                unassign.setToolTipText("Remove Diagram Template Assignment");
                createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, true);
                createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
            }
            @Override
            protected void buttonPressed(int buttonId) {
                super.buttonPressed(buttonId);
                if (buttonId == IDialogConstants.CLIENT_ID) {
                    setSelectionResult(new Object[0]);
                    setReturnCode(OK);
                    close();
                }
            }
            @Override
            protected IDialogSettings getDialogBoundsSettings() {
                return Activator.getDefault().getDialogSettings();
            }
        };
        dialog.setElements(elements);
        dialog.sortElements();
        dialog.setInitialSelection(initialSelection);
        dialog.setMultipleSelection(false);
        if (dialog.open() == Window.OK) {
            @SuppressWarnings("unchecked")
            Pair<Resource, String> result = (Pair<Resource, String>) dialog.getSingleResult();
            final Resource template = result != null ? result.first : null;
            Resource initialTemplate = initialSelection != null ? initialSelection.first : null;
            if (!Objects.equals(template, initialTemplate)) {
                Simantics.getSession().async(new WriteRequest() {
                    @Override
                    public void perform(WriteGraph graph) throws DatabaseException {
                        assignTemplateToDiagrams(graph, composites, template);
                    }
                });
            }
        }
    }

    private static class LabelProvider extends BaseLabelProvider implements ILabelProvider {

        @Override
        public Image getImage(Object element) {
            return null;
        }

        @SuppressWarnings("unchecked")
        @Override
        public String getText(Object element) {
            return ((Pair<?, String>) element).second;
        }

    }

    public static void assignTemplateToDiagrams(WriteGraph graph, Collection<Resource> composites, Resource template) throws DatabaseException {
        graph.markUndoPoint();
        
        ModelingResources MOD = ModelingResources.getInstance(graph);
        Template2dResource TEMPLATE2D = Template2dResource.getInstance(graph);

        StringBuilder comment = new StringBuilder();
        if (template != null) {
            comment.append("Assigned diagram template ").append(NameUtils.getSafeName(graph, template, true)).append(" to diagram");
        } else {
            comment.append("Removed diagram template from diagram");
        }
        if (composites.size() > 1)
            comment.append('s');
        comment.append('\n');

        for (Resource composite : composites) {
            Resource diagram = graph.getSingleObject(composite, MOD.CompositeToDiagram); 

            graph.deny(diagram, TEMPLATE2D.HasDrawingTemplate);
            if (template != null)
                graph.claim(diagram, TEMPLATE2D.HasDrawingTemplate, null, template);

            comment.append('\t').append(NameUtils.getSafeName(graph, composite, true));
        }

        CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
        graph.addMetadata( cm.add( comment.toString() ) );
    }

}
