package org.simantics.diagram.export;

import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

import org.simantics.NameLabelMode;
import org.simantics.NameLabelUtil;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.AssumptionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.exception.ValidationException;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.request.Read;
import org.simantics.export.core.ExportContext;
import org.simantics.export.core.error.ExportException;
import org.simantics.export.core.intf.ContentTypeAction;
import org.simantics.layer0.Layer0;
import org.simantics.simulation.ontology.SimulationResource;

public class DiagramContentTypeAction implements ContentTypeAction {
	
	public static String getDiagramLabel(ReadGraph graph, Resource r) throws DatabaseException
	{
		SimulationResource SIM = SimulationResource.getInstance(graph);

		NameLabelMode mode = NameLabelUtil.getNameLabelMode(graph);

		labelFailed:
   		try { 
			Resource model = graph.sync( new PossibleModel(r) );
			if ( model == null ) break labelFailed;
			Resource configuration = graph.getPossibleObject(model, SIM.HasConfiguration);
			if ( configuration == null ) break labelFailed;
			Deque<Resource> path = getPathTo(graph, r, model);
			if ( path == null) break labelFailed;
			path.remove(configuration);

			StringBuilder sb = new StringBuilder();
			for ( Resource node : path ) {
				if ( sb.length()>0 ) sb.append(" / ");
				String nodeLabel = NameLabelUtil.modalName(graph, node, mode);
				sb.append( nodeLabel );
			}
  				
			return sb.toString();
  		} catch (AssumptionException e) {
  		} catch (ValidationException e) {
   		} catch (ServiceException e) {
   		}
    		
		String uri = graph.getURI( r );
   		int c = uri.lastIndexOf('/');
   		if ( c>=0 ) {
   			String label = uri.substring(c+1);
   			if ( !label.isEmpty() ) return label;
   		}
   		
   		return uri;
	}

	@Override
	public Map<String, String> getLabels(ExportContext ctx, final Collection<String> uris) throws ExportException {
		Read<Map<String, String>> req = new UniqueRead<Map<String, String>>() {
			@Override
			public Map<String, String> perform(ReadGraph graph) throws DatabaseException {
				Map<String, String> result = new HashMap<String, String>();
				
				for ( String uri : uris ) {
       				Resource r = graph.getResource( uri );
					String label = getDiagramLabel(graph, r);
       				result.put(uri, label);
				}
				
				return result;
			}
		};
		
		try {
			return ctx.session.syncRequest( req );
		} catch (DatabaseException e) {
			throw new ExportException( e );
		}
	}
	
	/**
	 * Get all resource between start and end by following PartOf relation.
	 * If there is no path, returns null.
	 * Start is included, end is excluded.
	 * 
	 * @param graph
	 * @param start
	 * @param end
	 * @return path from end to start
	 * @throws ServiceException 
	 * @throws ManyObjectsForFunctionalRelationException 
	 */
	static Deque<Resource> getPathTo(ReadGraph graph, Resource start, Resource end) throws DatabaseException {
		LinkedList<Resource> result = new LinkedList<Resource>();
		
		Layer0 L0 = Layer0.getInstance(graph);
		Resource pos = start;
		while ( !pos.equals(end) ) {
			result.add(0, pos);
			pos = graph.getPossibleObject(pos, L0.PartOf);
			if ( pos==null ) return null;
		}
		return result;
	}

}
