/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.elements;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Path2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.EnumSet;
import java.util.Map;

import org.apache.commons.math3.geometry.euclidean.twod.Vector2D;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.g2d.canvas.Hints;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.SceneGraphNodeKey;
import org.simantics.g2d.element.handler.ElementHandler;
import org.simantics.g2d.element.handler.FillColor;
import org.simantics.g2d.element.handler.HandleMouseEvent;
import org.simantics.g2d.element.handler.InternalSize;
import org.simantics.g2d.element.handler.LifeCycle;
import org.simantics.g2d.element.handler.Move;
import org.simantics.g2d.element.handler.Outline;
import org.simantics.g2d.element.handler.Rotate;
import org.simantics.g2d.element.handler.Scale;
import org.simantics.g2d.element.handler.SceneGraph;
import org.simantics.g2d.element.handler.StaticSymbol;
import org.simantics.g2d.element.handler.Text;
import org.simantics.g2d.element.handler.TextEditor;
import org.simantics.g2d.element.handler.TextEditor.Modifier;
import org.simantics.g2d.element.handler.Transform;
import org.simantics.g2d.element.handler.impl.BorderColorImpl;
import org.simantics.g2d.element.handler.impl.FillColorImpl;
import org.simantics.g2d.element.handler.impl.ParentImpl;
import org.simantics.g2d.element.handler.impl.SimpleElementLayers;
import org.simantics.g2d.element.handler.impl.StaticSymbolImpl;
import org.simantics.g2d.element.handler.impl.TextColorImpl;
import org.simantics.g2d.element.handler.impl.TextEditorImpl;
import org.simantics.g2d.element.handler.impl.TextFontImpl;
import org.simantics.g2d.element.handler.impl.TextImpl;
import org.simantics.g2d.elementclass.MonitorHandler;
import org.simantics.g2d.image.Image;
import org.simantics.g2d.image.ProviderUtils;
import org.simantics.g2d.image.impl.AbstractImage;
import org.simantics.g2d.utils.Alignment;
import org.simantics.scenegraph.Node;
import org.simantics.scenegraph.g2d.G2DParentNode;
import org.simantics.scenegraph.g2d.events.MouseEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseEnterEvent;
import org.simantics.scenegraph.g2d.events.MouseEvent.MouseExitEvent;
import org.simantics.scenegraph.g2d.nodes.ShapeNode;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.utils.datastructures.cache.IFactory;
import org.simantics.utils.datastructures.cache.IProvider;
import org.simantics.utils.datastructures.cache.ProvisionException;
import org.simantics.utils.datastructures.hints.IHintContext.Key;
import org.simantics.utils.datastructures.hints.IHintContext.KeyOf;
import org.simantics.utils.strings.format.MetricsFormat;
import org.simantics.utils.strings.format.MetricsFormatList;

/**
 * @author Tuukka Lehtonen
 */
public class MonitorClass {

    static final Font        FONT                      = Font.decode("Helvetica 12");

    /**
     * Back-end specific object describing the monitored component.
     */
    public static final Key  KEY_MONITOR_COMPONENT     = new KeyOf(Object.class, "MONITOR_COMPONENT");

    /**
     * Existence of this hint indicates that the monitor is showing a value that
     * does not originate from the owner diagram.
     */
    public static final Key  KEY_MONITOR_IS_EXTERNAL   = new KeyOf(Boolean.class, "MONITOR_IS_EXTERNAL");

    /**
     * The valuation suffix string describing the monitored variable of the
     * component described by {@link #KEY_MONITOR_COMPONENT}.
     */
    public static final Key  KEY_MONITOR_SUFFIX        = new KeyOf(String.class, "MONITOR_SUFFIX");

    public static final Key  KEY_MONITOR_SUBSTITUTIONS = new KeyOf(Map.class, "MONITOR_SUBSTITUTIONS");
    public static final Key  KEY_MONITOR_GC            = new KeyOf(Graphics2D.class, "MONITOR_GC");
    public static final Key  KEY_MONITOR_HEIGHT        = new KeyOf(Double.class, "MONITOR_HEIGHT");
    public static final Key  KEY_NUMBER_FORMAT         = new KeyOf(MetricsFormat.class, "NUMBER_FORMAT");
    public static final Key  KEY_TOOLTIP_TEXT          = new KeyOf(String.class, "TOOLTIP_TEXT");

    public static final Key  KEY_EXPRESSION             = new KeyOf(String.class, "EXPRESSION");
    public static final Key  KEY_INPUT_VALIDATOR             = new KeyOf(Object.class, "INPUT_VALIDATOR");
    public static final Key  KEY_RVI             = new KeyOf(RVI.class, "RVI");

    public static final Key KEY_PREFIX_TEXT = new KeyOf(String.class, "PREFIX_TEXT");
    public static final Key KEY_SUFFIX_TEXT = new KeyOf(String.class, "SUFFIX_TEXT");

    /**
     * If this hint is defined, the monitor will force its x-axis to match this
     * angle. If this hint doesn't exist, the monitor will not force x-axis
     * orientation.
     */
    public static final Key  KEY_DIRECTION             = new KeyOf(Double.class, "MONITOR_DIRECTION");

    public static final Key  KEY_BORDER_WIDTH          = new KeyOf(Double.class, "MONITOR_BORDER");

    public static final Key  KEY_SG_NODE               = new SceneGraphNodeKey(TextNode.class, "MONITOR_SG_NODE");
    public static final Key  KEY_SG_NODE2              = new SceneGraphNodeKey(ShapeNode.class, "MONITOR_SG_NODE2");
    public static final Key KEY_SG_PREFIX_NODE         = new SceneGraphNodeKey(ShapeNode.class, "MONITOR_PREFIX_NODE");
    public static final Key KEY_SG_SUFFIX_NODE         = new SceneGraphNodeKey(ShapeNode.class, "MONITOR_SUFFIX_NODE");

    final static BasicStroke STROKE                    = new BasicStroke(1.0f);

    public final static Alignment DEFAULT_HORIZONTAL_ALIGN  = Alignment.CENTER;
    public final static Alignment DEFAULT_VERTICAL_ALIGN    = Alignment.CENTER;
    public final static MetricsFormat DEFAULT_NUMBER_FORMAT = MetricsFormatList.METRICS_DECIMAL;

    public final static Color     DEFAULT_FILL_COLOR        = new Color(224, 224, 224);
    public final static Color     DEFAULT_BORDER_COLOR      = Color.BLACK;

    public final static double    DEFAULT_HORIZONTAL_MARGIN = 5.0;
    public final static double    DEFAULT_VERTICAL_MARGIN   = 2.5;

    static Alignment getHorizontalAlignment(IElement e) {
        return ElementUtils.getHintOrDefault(e, ElementHints.KEY_HORIZONTAL_ALIGN, DEFAULT_HORIZONTAL_ALIGN);
    }

    static Alignment getVerticalAlignment(IElement e) {
        return ElementUtils.getHintOrDefault(e, ElementHints.KEY_VERTICAL_ALIGN, DEFAULT_VERTICAL_ALIGN);
    }

    static MetricsFormat getNumberFormat(IElement e) {
        return ElementUtils.getHintOrDefault(e, KEY_NUMBER_FORMAT, DEFAULT_NUMBER_FORMAT);
    }

    static void setNumberFormat(IElement e, MetricsFormat f) {
        ElementUtils.setOrRemoveHint(e, KEY_NUMBER_FORMAT, f);
    }

    static double getHorizontalMargin(IElement e) {
        return DEFAULT_HORIZONTAL_MARGIN;
    }

    static double getVerticalMargin(IElement e) {
        return DEFAULT_VERTICAL_MARGIN;
    }

    static Font getFont(IElement e) {
        return ElementUtils.getHintOrDefault(e, ElementHints.KEY_FONT, FONT);
    }

    public static class MonitorHandlerImpl implements MonitorHandler, HandleMouseEvent {
        private static final long          serialVersionUID = -4258875745321808416L;
        public static final MonitorHandler INSTANCE         = new MonitorHandlerImpl();
        @Override
        public boolean handleMouseEvent(IElement e, ICanvasContext ctx, MouseEvent me) {
            if (me instanceof MouseEnterEvent) {
                TextNode node = e.getHint(KEY_SG_NODE);
                if (node != null) {
                    node.setMouseOver(true);
                }
                return true;
            } else if (me instanceof MouseExitEvent) {
                TextNode node = e.getHint(KEY_SG_NODE);
                if (node != null) {
                    node.setMouseOver(false);
                }
                return false;
            }
            return false;
        }
    }

    static class Initializer implements LifeCycle {
        private static final long serialVersionUID = 4404942036933073584L;

        IElement parentElement;
        Map<String, String> substitutions;
        Object component;
        String suffix;
        boolean hack;

        Initializer(IElement parentElement, Map<String, String> substitutions, Object component, String suffix, boolean hack) {
            this.parentElement = parentElement;
            this.substitutions = substitutions;
            this.component = component;
            this.suffix = suffix;
            this.hack = hack;
        }

        @Override
        public void onElementActivated(IDiagram d, IElement e) {
            if(!hack) {
                hack = true;

                Point2D parentPos = ElementUtils.getPos(parentElement);
                Point2D thisPos = ElementUtils.getPos(e);

                Move move = e.getElementClass().getSingleItem(Move.class);
                move.moveTo(e, thisPos.getX() - parentPos.getX(), thisPos.getY() - parentPos.getY());
            }
        }
        @Override
        public void onElementCreated(IElement e) {
            if(parentElement != null) e.setHint(ElementHints.KEY_PARENT_ELEMENT, parentElement);
            if(substitutions != null) e.setHint(KEY_MONITOR_SUBSTITUTIONS, substitutions);
            if(component != null) e.setHint(KEY_MONITOR_COMPONENT, component);
            if(suffix != null) e.setHint(KEY_MONITOR_SUFFIX, suffix);
            e.setHint(KEY_DIRECTION, 0.0);
            e.setHint(KEY_NUMBER_FORMAT, DEFAULT_NUMBER_FORMAT);
            //e.setHint(KEY_HORIZONTAL_ALIGN, Alignment.LEADING);
            //e.setHint(KEY_VERTICAL_ALIGN, Alignment.LEADING);
        }

        @Override
        public void onElementDestroyed(IElement e) {
        }

        @Override
        public void onElementDeactivated(IDiagram d, IElement e) {
        }
    };

    static String finalText(IElement e) {
        String text = e.getElementClass().getSingleItem(Text.class).getText(e);
        if (text == null)
            return null;
        return substitute(text, e);
    }

    public static String editText(IElement e) {
        return substitute("#v1", e);
    }

    private static String formValue(IElement e) {
        // TODO: consider using substitute
        Map<String, String> substitutions = e.getHint(KEY_MONITOR_SUBSTITUTIONS);
        if (substitutions != null) {
            String value = substitutions.get("#v1");
            if (substitutions.containsKey("#u1") && substitutions.get("#u1").length() > 0) {
                value += " " + substitutions.get("#u1");
            }
            return value;
        } else {
        	return ElementUtils.getText(e);
        }
    }

    static String substitute(String text, IElement e) {
        Map<String, String> substitutions = e.getHint(KEY_MONITOR_SUBSTITUTIONS);
        return substitute(text, substitutions);
    }

    static String substitute(String text, Map<String, String> substitutions) {
        if (substitutions != null) {
            // TODO: slow as hell
            for(Map.Entry<String, String> entry : substitutions.entrySet()) {
                if (entry.getValue() != null) {
                    text = text.replace(entry.getKey(), entry.getValue());
                } else {
                    text = text.replace(entry.getKey(), "<null>");
                }
            }
        }
        return text;
    }

    public static void update(IElement e) {
        MonitorSGNode node = e.getElementClass().getSingleItem(MonitorSGNode.class);
        node.update(e);
    }

    public static void cleanup(IElement e) {
        MonitorSGNode node = e.getElementClass().getSingleItem(MonitorSGNode.class);
        node.cleanup(e);
    }

    public static boolean hasModifier(IElement e) { 
        TextEditor editor = e.getElementClass().getAtMostOneItemOfClass(TextEditor.class);
        Modifier modifier = editor != null ? editor.getModifier(e) : null;
        return modifier != null;
    }

    static final Rectangle2D DEFAULT_BOX = new Rectangle2D.Double(0, 0, 0, 0);

    static Shape createMonitor(IElement e) {
        Alignment hAlign = getHorizontalAlignment(e);
        Alignment vAlign = getVerticalAlignment(e);
        double hMargin = getHorizontalMargin(e);
        double vMargin = getVerticalMargin(e);

        String text = finalText(e);
        if(text == null) {
            return align(hMargin, vMargin, hAlign, vAlign, DEFAULT_BOX);
        }

        Graphics2D g = e.getHint(KEY_MONITOR_GC);
        if(g == null) {
            return align(hMargin, vMargin, hAlign, vAlign, DEFAULT_BOX);
        }

        Font f = getFont(e);
        FontMetrics fm   = g.getFontMetrics(f);
        Rectangle2D rect = fm.getStringBounds(text, g);

        return align(hMargin, vMargin, hAlign, vAlign, rect);
    }

    static Shape align(double hMargin, double vMargin, Alignment hAlign, Alignment vAlign, Rectangle2D rect) {
        //System.out.println("align: " + hMargin + ", " + vMargin + ", " + hAlign + ", " + vAlign + ": " + rect);
        double tx = align(hMargin, hAlign, rect.getMinX(), rect.getMaxX());
        double ty = align(vMargin, vAlign, rect.getMinY(), rect.getMaxY());
        //System.out.println("    translate: " + tx + " "  + ty);
        double nw = rect.getWidth() + 2*hMargin;
        double nh = rect.getHeight() + 2*vMargin;
        return makePath(tx + rect.getMinX(), ty + rect.getMinY(), nw, nh);
    }

    static double align(double margin, Alignment align, double min, double max) {
        double s = max - min;
        switch (align) {
            case LEADING:
                return -min;
            case TRAILING:
                return -s - 2 * margin - min;
            case CENTER:
                return -0.5 * s - margin - min;
            default:
                return 0;
        }
    }

    static Path2D makePath(double x, double y, double w, double h) {
        Path2D path = new Path2D.Double();
        path.moveTo(x, y);
        path.lineTo(x+w, y);
        path.lineTo(x+w, y+h);
        path.lineTo(x, y+h);
        path.closePath();
        return path;
    }

    public static final Shape BOX_SHAPE = new Rectangle(-1, -1, 2, 2);

    public static class MonitorSGNode implements SceneGraph, InternalSize, Outline {
        private static final long serialVersionUID = -106278359626957687L;

        static final MonitorSGNode INSTANCE = new MonitorSGNode();

        @SuppressWarnings("unchecked")
        @Override
        public void init(final IElement e, final G2DParentNode parent) {
            // Create node if it doesn't exist yet
            TextNode node = e.getHint(KEY_SG_NODE);
            String nodeId = null;
            if(node == null || node.getBounds() == null || node.getParent() != parent) {
                nodeId = ElementUtils.generateNodeId(e);
                node = parent.addNode(nodeId, TextNode.class);
                node.setElementBasedFocus(true);
                e.setHint(KEY_SG_NODE, node);

                node.setTextListener(new ITextListener() {

                    @Override
                    public void textChanged() {
                        updatePrefixAndSuffixPositions(e);
                        e.setHint(Hints.KEY_DIRTY, Hints.VALUE_SG_DIRTY);
                    }

                    @Override
                    public void textEditingStarted() {
                        updatePrefixAndSuffixPositions(e);
                    }

                    @Override
                    public void textEditingCancelled() {
                        updatePrefixAndSuffixPositions(e);
                    }

                    boolean isEndingEdit = false;

                    @Override
                    public void textEditingEnded() {
                        updatePrefixAndSuffixPositions(e);
                        TextNode node = e.getHint(KEY_SG_NODE);
                        if (node == null)
                            return;

                        // Prevent recursive execution which will happen
                        // due to the endEdit(node) invocation at the end.
                        if (isEndingEdit)
                            return;
                        isEndingEdit = true;

                        try {
                            TextEditor editor = e.getElementClass().getAtMostOneItemOfClass(TextEditor.class);
                            if (editor != null) {
                                Modifier modifier = editor.getModifier(e);
                                if (modifier != null) {
                                    String newValue = node.getText();
                                    String error = modifier.isValid(e, newValue);
                                    if (error == null) {
                                        // Only modify if the modification was not
                                        // cancelled and the value is valid.
                                        modifier.modify(e, newValue);
                                    } else {
                                        // TODO: show error somehow, possibly through status bar

                                        // Make sure that the monitor content gets
                                        // reset to its previous value.
                                        node.setText(formValue(e));
                                    }
                                }
                            }
                        } finally {
                            isEndingEdit = false;
                        }
                    }
                });

                Object validator = e.getHint(KEY_INPUT_VALIDATOR);
                if(validator != null) {
                    node.setValidator((Function1<String, String>)validator);
                }
                
                RVI rvi = e.getHint(KEY_RVI);
                if(rvi != null) {
                    node.setRVI(rvi);
                }

                Double border_width = (Double)e.getHint(KEY_BORDER_WIDTH);
                if(border_width == null) border_width = 0.1;

                node.setBorderWidth(border_width);

//                Rectangle2D bounds = (Rectangle2D)e.getHint(ElementHints.KEY_BOUNDS);
//                if(bounds != null) node.setBounds(bounds);
                Font font = ElementUtils.getTextFont(e);
                Color color = ElementUtils.getTextColor(e);
                String text = ElementUtils.getText(e);
                node.init(text, font, color, 0, 0, 1.0);
            }

            Boolean isExternal = e.getHint(KEY_MONITOR_IS_EXTERNAL);
            if (Boolean.TRUE.equals(isExternal)) {
                ShapeNode shape = e.getHint(KEY_SG_NODE2);
                if (shape == null || shape.getParent() != parent) {
                    if (nodeId == null)
                        nodeId = ElementUtils.generateNodeId(e);
                    nodeId += "-ext";
                    shape = parent.addNode(nodeId, ShapeNode.class);
                    shape.setZIndex(-1);
                    shape.setColor(Color.BLACK);
                    shape.setFill(true);
                    shape.setStroke(null);
                    shape.setShape( arrow(4, 2, 0) );
                    e.setHint(KEY_SG_NODE2, shape);
                }
            } else {
                ShapeNode shape = e.getHint(KEY_SG_NODE2);
                if (shape != null)
                    shape.remove();
            }

            String prefixText = e.getHint(KEY_PREFIX_TEXT);
            if (prefixText != null) {
                TextNode prefixNode = e.getHint(KEY_SG_PREFIX_NODE);
                if (prefixNode == null || prefixNode.getParent() != parent) {
                    nodeId = ElementUtils.generateNodeId(e) + "-prefix";
                    node = parent.addNode(nodeId, TextNode.class);
                    e.setHint(KEY_SG_PREFIX_NODE, node);
                    Font font = ElementUtils.getTextFont(e);
                    Color color = ElementUtils.getTextColor(e);
                    node.setEditable(false);
                    node.setShowSelection(false);
                    node.setHorizontalAlignment((byte) 1);
                    node.init(prefixText, font, color, 0, 0, 1.0);
                }
            } else {
                TextNode prefixNode = e.getHint(KEY_SG_PREFIX_NODE);
                if (prefixNode != null)
                    prefixNode.remove();
            }

            String suffixText = e.getHint(KEY_SUFFIX_TEXT);
            if (suffixText != null) {
                TextNode suffixNode = e.getHint(KEY_SG_SUFFIX_NODE);
                if (suffixNode == null || suffixNode.getParent() != parent) {
                    nodeId = ElementUtils.generateNodeId(e) + "-suffix";
                    node = parent.addNode(nodeId, TextNode.class);
                    e.setHint(KEY_SG_SUFFIX_NODE, node);
                    Font font = ElementUtils.getTextFont(e);
                    Color color = ElementUtils.getTextColor(e);
                    node.setEditable(false);
                    node.setShowSelection(false);
                    node.setHorizontalAlignment((byte) 0);
                    node.init(suffixText, font, color, 0, 0, 1.0);
                }
            } else {
                TextNode suffixNode = e.getHint(KEY_SG_SUFFIX_NODE);
                if (suffixNode != null)
                    suffixNode.remove();
            }

            update(e);
        }

        public void update(IElement e) {
            String value = null;

            final Text t = e.getElementClass().getAtMostOneItemOfClass(Text.class);
            assert(t != null);

            value = formValue(e);

            TextNode node = (TextNode) e.getHint(KEY_SG_NODE);
            if (node != null && value != null) {

                node.setText(value);
                Object component = e.getHint(KEY_MONITOR_COMPONENT);
                if (component != null && hasModifier(e)) {
                    node.setEditable(true);
                } else {
                    node.setEditable(false);
                }

                // FIXME: set only if changed .. (but quickfix is not to clone)
                Font font = ElementUtils.getTextFont(e);
                if (node.getFont() != font) { // Don't update if we have a same object
                    node.setFont(font);
                }
                Color color = ElementUtils.getTextColor(e);
                node.setColor(new Color(color.getRed(), color.getGreen(), color.getBlue(), color.getAlpha()));

                AffineTransform at = ElementUtils.getTransform(e);
                if(at != null)
                    node.setTransform(at);

                Alignment halign = e.getHint(ElementHints.KEY_HORIZONTAL_ALIGN);
                if (halign != null)
                    node.setHorizontalAlignment((byte) halign.ordinal());
                Alignment valign = e.getHint(ElementHints.KEY_VERTICAL_ALIGN);
                if (valign != null)
                    node.setVerticalAlignment((byte) valign.ordinal());

//                String tt = (String) e.getHint(KEY_TOOLTIP_TEXT);
//                if (tt != null)
//                    node.setToolTipText(new String(tt));

                ShapeNode shape = e.getHint(KEY_SG_NODE2);
                if (shape != null) {
                    AffineTransform at2 = new AffineTransform(at);
                    Rectangle2D bounds = node.getBoundsInLocal();
                    at2.translate(bounds.getMinX(), bounds.getCenterY());
                    shape.setTransform(at2);
                }

                TextNode prefixNode = e.getHint(KEY_SG_PREFIX_NODE);
                if (prefixNode != null) {
                    if (prefixNode.getFont() != font) {
                        prefixNode.setFont(font);
                    }
                    prefixNode.setColor(color);
                }
                TextNode suffixNode = e.getHint(KEY_SG_SUFFIX_NODE);
                if (suffixNode != null) {
                    if (suffixNode.getFont() != font) {
                        suffixNode.setFont(font);
                    }
                    suffixNode.setColor(color);
                }

                updatePrefixAndSuffixPositions(e);
            }
        }

        public void updatePrefixAndSuffixPositions(IElement e) {
            TextNode node = e.getHint(KEY_SG_NODE);
            if (node != null) {
                TextNode prefix = e.getHint(KEY_SG_PREFIX_NODE);
                TextNode suffix = e.getHint(KEY_SG_SUFFIX_NODE);
                if (prefix != null || suffix != null) {
                    AffineTransform at = ElementUtils.getTransform(e);
                    Alignment vAlign = getVerticalAlignment(e);
                    if (prefix != null) {
                        AffineTransform at2 = new AffineTransform(at);
                        Rectangle2D bounds = node.getBoundsInLocal();
                        at2.translate(bounds.getMinX(), 0);
                        prefix.setTransform(at2);
                        prefix.setVerticalAlignment((byte) vAlign.ordinal());
                    }

                    if (suffix != null) {
                        AffineTransform at2 = new AffineTransform(at);
                        Rectangle2D bounds = node.getBoundsInLocal();
                        at2.translate(bounds.getMaxX(), 0);
                        suffix.setTransform(at2);
                        suffix.setVerticalAlignment((byte) vAlign.ordinal());
                    }
                }
            }
        }

        @Override
        public void cleanup(IElement e) {
            TextNode node = (TextNode)e.removeHint(KEY_SG_NODE);
            if (node != null)
                node.remove();
            TextNode prefixNode = (TextNode) e.getHint(KEY_SG_PREFIX_NODE);
            if (prefixNode != null)
                prefixNode.remove();
            TextNode suffixNode = (TextNode) e.getHint(KEY_SG_SUFFIX_NODE);
            if (suffixNode != null)
                suffixNode.remove();
        }

        @Override
        public Rectangle2D getBounds(IElement e, Rectangle2D size) {
            TextNode node = (TextNode)e.getHint(KEY_SG_NODE);
            if (node != null) {
                Rectangle2D bounds = node.getBoundsInLocal();
                if (bounds != null) {
                    if (size == null)
                        size = new Rectangle2D.Double(0, 0, 0, 0);
                    size.setRect(bounds);
                }
            }
            TextNode prefixNode = (TextNode) e.getHint(KEY_SG_PREFIX_NODE);
            if (prefixNode != null) {
                Rectangle2D bounds = node.parentToLocal(prefixNode.localToParent(prefixNode.getBoundsInLocal()));
                if (size == null)
                    size = new Rectangle2D.Double(0, 0, 0, 0);
                size.add(bounds);
            }
            TextNode suffixNode = (TextNode) e.getHint(KEY_SG_SUFFIX_NODE);
            if (suffixNode != null) {
                Rectangle2D bounds = node.parentToLocal(suffixNode.localToParent(suffixNode.getBoundsInLocal()));
                if (size == null)
                    size = new Rectangle2D.Double(0, 0, 0, 0);
                size.add(bounds);
            }
            return size;
        }

        @Override
        public Shape getElementShape(IElement e) {
            Rectangle2D bounds = getBounds(e, null);
            if (bounds != null) {
                return bounds;
            } else {
                return new Rectangle2D.Double(0, 0, 0, 0);
            }
        }

    }

    public static class Transformer implements Transform, Move, Rotate, Scale, LifeCycle {

        private static final long serialVersionUID = -3704887325602085677L;

        public static final Transformer INSTANCE = new Transformer(null);

        Double aspectRatio;

        public Transformer() {
            this(null);
        }

        public Transformer(Double aspectRatio) {
            this.aspectRatio = aspectRatio;
        }

        @Override
        public Double getFixedAspectRatio(IElement e) {
            return aspectRatio;
        }

        @Override
        public Point2D getScale(IElement e) {
            AffineTransform at = e.getHint(ElementHints.KEY_TRANSFORM);
            return _getScale(at);
        }

        @Override
        public void setScale(IElement e, Point2D newScale) {
            // Doesn't work for monitors.
            Point2D oldScale = getScale(e);
            double sx = newScale.getX() / oldScale.getX();
            double sy = newScale.getY() / oldScale.getY();
            AffineTransform at = e.getHint(ElementHints.KEY_TRANSFORM);
            at = new AffineTransform(at);
            at.scale(sx, sy);
            e.setHint(ElementHints.KEY_TRANSFORM, at);
        }

        @Override
        public Point2D getMaximumScale(IElement e) {
            return null;
        }

        @Override
        public Point2D getMinimumScale(IElement e) {
            return null;
        }

        private static Point2D _getScale(AffineTransform at) {
            double m00 = at.getScaleX();
            double m11 = at.getScaleY();
            double m10 = at.getShearY();
            double m01 = at.getShearX();
            // Project unit vector to canvas
            double sx = Math.sqrt(m00 * m00 + m10 * m10);
            double sy = Math.sqrt(m01 * m01 + m11 * m11);
            return new Point2D.Double(sx, sy);
        }

        @Override
        public void rotate(IElement e, double theta, Point2D origin) {
            if (Double.isNaN(theta)) return;
            theta = Math.toDegrees(theta);
            Double angle = e.getHint(KEY_DIRECTION);
            double newAngle = angle != null ? angle+theta : theta;
            newAngle = Math.IEEEremainder(newAngle, 360.0);
            e.setHint(KEY_DIRECTION, newAngle);
        }

        @Override
        public double getAngle(IElement e) {
            Double angle = e.getHint(KEY_DIRECTION);
            return angle != null ? Math.toRadians(angle) : 0;
        }
        @Override
        public Point2D getPosition(IElement e) {
            AffineTransform at = e.getHint(ElementHints.KEY_TRANSFORM);
            Point2D p = new Point2D.Double(at.getTranslateX(), at.getTranslateY());
            return p;
        }

        @Override
        public void moveTo(IElement e, double x, double y) {
            AffineTransform origAt = e.getHint(ElementHints.KEY_TRANSFORM);

            AffineTransform result = new AffineTransform(origAt);
            result.preConcatenate(AffineTransform.getTranslateInstance(x - origAt.getTranslateX(), y - origAt.getTranslateY()));
            e.setHint(ElementHints.KEY_TRANSFORM, result);
        }

        @Override
        public AffineTransform getTransform(IElement e) {
            AffineTransform at = e.getHint(ElementHints.KEY_TRANSFORM);

            IElement parentElement = e.getHint(ElementHints.KEY_PARENT_ELEMENT);
            if (parentElement == null)
                return at;

            Transform parentTransform = parentElement.getElementClass().getSingleItem(Transform.class);
            assert(parentTransform!=null);

            AffineTransform result = (AffineTransform)at.clone();
            AffineTransform parentAT = parentTransform.getTransform(parentElement);
            result.preConcatenate(AffineTransform.getTranslateInstance(parentAT.getTranslateX(), parentAT.getTranslateY()));

            return result;
        }

        @Override
        public void setTransform(IElement e, AffineTransform at) {
            e.setHint(ElementHints.KEY_TRANSFORM, at.clone());
        }

        @Override
        public void onElementActivated(IDiagram d, IElement e) {
        }

        @Override
        public void onElementCreated(IElement e) {
            e.setHint(ElementHints.KEY_TRANSFORM, new AffineTransform());
        }

        @Override
        public void onElementDeactivated(IDiagram d, IElement e) {
        }

        @Override
        public void onElementDestroyed(IElement e) {
//            List<SceneGraph> nodeHandlers = e.getElementClass().getItemsByClass(SceneGraph.class);
//            for(SceneGraph n : nodeHandlers) {
//                System.out.println("element gone:"+e);
//                n.cleanup(e);
//            }
        }
    }

    static double getOrientationDelta(IElement e, AffineTransform tr) {
        Double angle = e.getHint(KEY_DIRECTION);
        if (angle == null || Double.isNaN(angle))
            return Double.NaN;
        double angrad = Math.toRadians(angle);

        Vector2D forcedAxis = new Vector2D(Math.cos(angrad), Math.sin(angrad));
        Vector2D x = new Vector2D(tr.getScaleX(), tr.getShearX()).normalize();
        double cosa = forcedAxis.dotProduct(x);
        double delta = Math.acos(cosa);
        return delta;
    }

    static class MonitorImageFactory implements IFactory<Image> {
        private double staticScaleX = 1, staticScaleY = 1;

        public MonitorImageFactory(double staticScaleX, double staticScaleY) {
            this.staticScaleX = staticScaleX;
            this.staticScaleY = staticScaleY;
        }

        @Override
        public Image get() throws ProvisionException {
            return new Img();
        }

        public class Img extends AbstractImage {

            Shape path = align(DEFAULT_HORIZONTAL_MARGIN, DEFAULT_VERTICAL_MARGIN, DEFAULT_HORIZONTAL_ALIGN, DEFAULT_VERTICAL_ALIGN,
                    new Rectangle2D.Double(0, 0, 50*staticScaleX, 22*staticScaleY));

            @Override
            public int hashCode() {
                long temp = Double.doubleToLongBits(staticScaleX);
                int result = (int) (temp ^ (temp >>> 32));
                temp = Double.doubleToLongBits(staticScaleY);
                result = 31 * result + (int) (temp ^ (temp >>> 32));
                return result;
            }

            @Override
            public boolean equals(Object obj) {
                if (this == obj)
                    return true;
                if (obj == null)
                    return false;
                if (getClass() != obj.getClass())
                    return false;
                MonitorImageFactory other = (MonitorImageFactory) obj;
                return Double.doubleToLongBits(staticScaleX) == Double.doubleToLongBits(other.staticScaleX)
                        && Double.doubleToLongBits(staticScaleY) != Double.doubleToLongBits(other.staticScaleY);
            }

            @Override
            public Rectangle2D getBounds() {
                return path.getBounds2D();
            }

            @Override
            public EnumSet<Feature> getFeatures() {
                return EnumSet.of(Feature.Vector);
            }

            @Override
            public Shape getOutline() {
                return path;
            }

            @Override
            public Node init(G2DParentNode parent) {
                TextNode node = parent.getOrCreateNode(""+hashCode(), TextNode.class);
                node.init("Drop Me", FONT, Color.BLACK, 0, 0, 0.2);
                //node.setSize(50, 22);
                node.setBorderWidth(1);
                node.setTransform(AffineTransform.getScaleInstance(staticScaleX, staticScaleY));
                node.setEditable(false);
                return node;
            }
        };
    }

    static final IProvider<Image> MONITOR_IMAGE =
        ProviderUtils.reference(
                ProviderUtils.cache(
                        ProviderUtils.rasterize(
                                new MonitorImageFactory(0.5, 0.5)
                        )));

    static final StaticSymbol MONITOR_SYMBOL = new StaticSymbolImpl( MONITOR_IMAGE.get() );

    static final FillColor FILL_COLOR = new FillColorImpl(DEFAULT_FILL_COLOR);

    static final ElementClass MONITOR_CLASS_BASE =
        ElementClass.compile(
                MonitorHandlerImpl.INSTANCE,
                Transformer.INSTANCE,
                BorderColorImpl.BLACK,
                FILL_COLOR,
                MonitorSGNode.INSTANCE,
//                ClickableImpl.INSTANCE,
                TextImpl.INSTANCE,
                TextEditorImpl.INSTANCE,
                TextFontImpl.DEFAULT,
                TextColorImpl.BLACK,
                SimpleElementLayers.INSTANCE,
                ParentImpl.INSTANCE
        );

    // staticScale{X,Y} define the scale of the static monitor image
    public static ElementClass create(double staticScaleX, double staticScaleY, ElementHandler... extraHandlers) {
        // Bit of a hack to be able to define the scale
        IProvider<Image> staticMonitorSymbolProvider = ProviderUtils.reference(
                ProviderUtils.cache(
                        ProviderUtils
                        .rasterize(
                                new MonitorImageFactory(staticScaleX, staticScaleY))));
        StaticSymbol staticMonitorSymbol = new StaticSymbolImpl( staticMonitorSymbolProvider.get() );
        return MONITOR_CLASS_BASE.newClassWith(staticMonitorSymbol).newClassWith(extraHandlers);
    }

    // staticScale{X,Y} define the scale of the static monitor image
    public static ElementClass create(IElement parentElement, Map<String, String> substitutions, Object component, String suffix, double staticScaleX, double staticScaleY, ElementHandler... extraHandlers) {
        return create(staticScaleX, staticScaleY, extraHandlers)
                .newClassWith(
                        new Initializer(parentElement, substitutions, component,
                                suffix, parentElement != null ? false : true));
    }

    public static Shape arrow(double length, double width, double space) {
        Path2D.Double path = new Path2D.Double();
        path.moveTo(-space, 0);
        path.lineTo(-length - space, -width);
        path.lineTo(-length - space, +width);
        path.closePath();
        return path;
    }

}
