///////////////////////////////////////////////////////
//                                                   //
//   VTT Technical Research Centre of Finland LTD    //
//   For internal use only. Do not redistribute.     //
//                                                   //
//   Authors:                                        //
//       Antton Tapani    ext-antton.tapani@vtt.fi   //
//                                                   //
//   Last modified by Antton Tapani    9.2016        //
//                                                   //
///////////////////////////////////////////////////////

#include <windows.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <direct.h>
#include "engine.h"
#include <jni.h>

#pragma warning(disable: 4996)

#define UTF8FEATURE "feature('DefaultCharacterSet', 'UTF-8')"

#define BUFSIZE 1000

#define MAXJSIZE MAXLONG
#define MAXJINT MAXLONG

#define tojsize(x) ((jsize)(min((x), MAXJSIZE)))
#define tojint(x) ((jint)(min((x), MAXJINT)))

jobject getJavaStructArray(JNIEnv *env, mxArray *matlabArray);
jobject getJavaCharacterArray(JNIEnv *env, mxArray *matlabArray);
jobject getJavaDoubleArray(JNIEnv *env, mxArray *matlabArray);
jobject getJavaCellArray(JNIEnv *env, mxArray *matlabArray);
jobject getJavaArray(JNIEnv *env, mxArray *item);

mxArray *getMatlabArray(JNIEnv *env, jobject item);
mxArray *getMatlabDoubleArray(JNIEnv *env, jobject value);
mxArray *getMatlabCharacterArray(JNIEnv *env, jobject value);
mxArray *getMatlabStructArray(JNIEnv *env, jobject value);
mxArray *getMatlabCellArray(JNIEnv *env, jobject value);

jint throwException( JNIEnv *env, char *className, char *message )
{
    jclass exClass = (*env)->FindClass( env, className);
    if (exClass == NULL) {
        return 0;
    }

    return (*env)->ThrowNew( env, exClass, message );
}

jint throwIllegalArgumentException( JNIEnv *env, char *message ) {
	return throwException( env, "java/lang/IllegalArgumentException", message );
}

/**
 * Get the dimensions of a Matlab array as a Java integer[].
 * The value pointed to by nelems is set to the total number of elements int
 * the array.
 */
jintArray getMatlabArrayDims(JNIEnv *env, mxArray *matlabArray) {
	int i;

	size_t ndims = mxGetNumberOfDimensions(matlabArray);
	const size_t *dims = mxGetDimensions(matlabArray);

	jintArray jdims = (*env)->NewIntArray(env, tojsize(ndims));

	for (i = 0; i < ndims; i++) {
		jsize dim = tojsize(dims[i]);
		(*env)->SetIntArrayRegion(env, jdims, i, 1, &dim);
	}

	return jdims;
}

/**
 * Get the contents of a Matlab double array as (Java) double[].
 */
jobject getJavaDoubleArray(JNIEnv *env, mxArray *matlabArray) {
	jclass doubleArrayClass = (*env)->FindClass(env, "org/simantics/matlablink/DoubleArray");
	jmethodID constructor = (*env)->GetMethodID(env, doubleArrayClass, "<init>", "([I[D)V");

	jsize nelems = tojsize(mxGetNumberOfElements(matlabArray));
	jdoubleArray values = (*env)->NewDoubleArray(env, nelems);
	double *contents = mxGetPr(matlabArray);
	(*env)->SetDoubleArrayRegion(env, values, 0, nelems, contents);

	{
		jintArray jdims = getMatlabArrayDims(env, matlabArray);
		jobject result = (*env)->NewObject(env, doubleArrayClass, constructor, jdims, values);
		return result;
	}
}

/**
 * Get the contents of a Matlab character array as java.lang.String.
 */
jobject getJavaCharacterArray(JNIEnv *env, mxArray *matlabArray) {
	jclass characterArrayClass = (*env)->FindClass(env, "org/simantics/matlablink/CharacterArray");
	jmethodID constructor = (*env)->GetMethodID(env, characterArrayClass, "<init>", "([I[C)V");

	{
		size_t len = mxGetNumberOfElements(matlabArray);
		mxChar *buf = mxGetChars(matlabArray);
		jintArray jdims = getMatlabArrayDims(env, matlabArray);
		jcharArray bytes = (*env)->NewCharArray(env, tojsize(len));
		(*env)->SetCharArrayRegion(env, bytes, 0, tojsize(len), buf);
		{
			jobject result = (*env)->NewObject(env, characterArrayClass, constructor, jdims, bytes);
			return result;
		}
	}
}

/**
 * Get the contents of a Matlab struct array as java.util.Map[].
 */
jobject getJavaStructArray(JNIEnv *env, mxArray *matlabArray) {
	jclass structArrayClass = (*env)->FindClass(env, "org/simantics/matlablink/StructArray");
	jmethodID constructor = (*env)->GetMethodID(env, structArrayClass, "<init>", "([ILjava/util/Collection;)V");
	jmethodID setFieldMethod = (*env)->GetMethodID(env, structArrayClass, "setField", "(ILjava/lang/String;Lorg/simantics/matlablink/MatlabArray;)V");

	jclass arrayListClass = (*env)->FindClass(env, "java/util/ArrayList");
	jmethodID arrayListConstructor = (*env)->GetMethodID(env, arrayListClass, "<init>", "()V");
	jmethodID addMethod = (*env)->GetMethodID(env, arrayListClass, "add", "(Ljava/lang/Object;)Z");
	jmethodID getMethod = (*env)->GetMethodID(env, arrayListClass, "get", "(I)Ljava/lang/Object;");

	size_t nelems = mxGetNumberOfElements(matlabArray);
	jintArray jdims = getMatlabArrayDims(env, matlabArray);
	size_t len = mxGetNumberOfElements(matlabArray);
	int nFields = mxGetNumberOfFields(matlabArray);
	jsize jlen = tojsize(len);

	jobject fields = (*env)->NewObject(env, arrayListClass, arrayListConstructor);

	int j;

	for (j = 0; j < nFields; j++) {
		jstring fieldName = (*env)->NewStringUTF(env, mxGetFieldNameByNumber(matlabArray, j));
		(*env)->CallVoidMethod(env, fields, addMethod, fieldName);
	}

	{
		jobject result = (*env)->NewObject(env, structArrayClass, constructor, jdims, fields);

		for (j = 0; j < nFields; j++) {
			jstring fieldName = (jstring)(*env)->CallObjectMethod(env, fields, getMethod, j);
			jint i;
			for (i = 0; i < jlen; i++) {
				jobject value = getJavaArray(env, mxGetFieldByNumber(matlabArray, i, j));
				(*env)->CallVoidMethod(env, result, setFieldMethod, i, fieldName, value);
			}
		}

		return result;
	}
}

/**
 * Get the contents of a Matlab cell array as java.lang.Object[].
 */
jobject getJavaCellArray(JNIEnv *env, mxArray *matlabArray) {
	size_t i;
	jclass cellArrayClass = (*env)->FindClass(env, "org/simantics/matlablink/CellArray");
	jmethodID constructor = (*env)->GetMethodID(env, cellArrayClass, "<init>", "([I)V");
	jmethodID setCellItemMethod = (*env)->GetMethodID(env, cellArrayClass, "setCellItem", "(ILorg/simantics/matlablink/MatlabArray;)V");

	jintArray jdims = getMatlabArrayDims(env, matlabArray);
	jobject result = (*env)->NewObject(env, cellArrayClass, constructor, jdims);

	size_t len = mxGetNumberOfElements(matlabArray);
	for (i = 0; i < len; i++) {
		mxArray *item = mxGetCell(matlabArray, i);
		if (item != NULL)
			(*env)->CallVoidMethod(env, result, setCellItemMethod, (jint)i, getJavaArray(env, item));
	}

	return result;
}

/**
 * Get the value of a Matlab array as a type-dependent Java object.
 *
 * The currently supported Matlab data types are:
 *  - character
 *  - double
 *  - struct
 *  - cell
 */
jobject getJavaArray(JNIEnv *env, mxArray *item) {
	switch (mxGetClassID(item)) {
	case mxCHAR_CLASS:
		return getJavaCharacterArray(env, item);
		break;
	case mxCELL_CLASS:
		return getJavaCellArray(env, item);
		break;
	case mxSTRUCT_CLASS:
		return getJavaStructArray(env, item);
		break;
	case mxDOUBLE_CLASS:
		return getJavaDoubleArray(env, item);
		break;
	default:
		return NULL;
		break;
	}
}

JNIEXPORT jlong JNICALL Java_org_simantics_matlablink_Engine_openMatlabEngineImpl(JNIEnv *env, jobject thisObj, jstring workingDirectory) {
	jsize len = (*env)->GetStringUTFLength(env, workingDirectory);
	const char *bytes = (*env)->GetStringUTFChars(env, workingDirectory, NULL);
	char *buf = (char*)malloc(len + 6);

	// Open Matlab engine
	Engine *engine = engOpen(NULL);
	jlong result = (jlong)engine;

	// Make Matlab use UTF-8 by default
	engEvalString(engine, UTF8FEATURE);

	// Switch current directory
	sprintf(buf, "cd('%s')", bytes);
	(*env)->ReleaseStringUTFChars(env, workingDirectory, bytes);
	engEvalString(engine, buf);

	// Return the engine pointer as a Java long
	return result;
}

JNIEXPORT void JNICALL Java_org_simantics_matlablink_Engine_closeMatlabEngineImpl(JNIEnv *env, jobject thisObj, jlong engineID) {
	Engine *engine = (Engine *)engineID;
	engClose(engine);
}

mxArray *getMatlabVariable(JNIEnv *env, Engine *engine, jstring variableName) {
	const char *name = (*env)->GetStringUTFChars(env, variableName, NULL);
	mxArray *result = engGetVariable(engine, name);
	(*env)->ReleaseStringUTFChars(env, variableName, name);
	return result;
}

int putMatlabVariable(JNIEnv *env, Engine *engine, jstring variableName, mxArray *value) {
	const char *name = (*env)->GetStringUTFChars(env, variableName, NULL);
	int status = engPutVariable(engine, name, value);
	(*env)->ReleaseStringUTFChars(env, variableName, name);
	return status;
}

mxArray *getMatlabDoubleArray(JNIEnv *env, jobject javaObject) {
	jclass doubleArrayClass = (*env)->FindClass(env, "org/simantics/matlablink/DoubleArray");
	jmethodID getDoubleValueMethod = (*env)->GetMethodID(env, doubleArrayClass, "getDoubleValue", "()[D");
	jmethodID dimsMethod = (*env)->GetMethodID(env, doubleArrayClass, "dims", "()[I");

	jdoubleArray value = (*env)->CallObjectMethod(env, javaObject, getDoubleValueMethod);
	jintArray jdims = (*env)->CallObjectMethod(env, javaObject, dimsMethod);

	// Get vector dimentions
	jsize ndims = (*env)->GetArrayLength(env, jdims);
	jsize *jdimArray = (*env)->GetIntArrayElements(env, jdims, NULL);
	size_t *dims = (size_t*)malloc(ndims * sizeof(size_t));

	jint i;

	for (i = 0; i < ndims; i++) {
		dims[i] = jdimArray[i];
	}

	(*env)->ReleaseIntArrayElements(env, jdims, jdimArray, JNI_ABORT);

	{
		mxArray *result = mxCreateNumericArray(ndims, dims, mxDOUBLE_CLASS, mxREAL);
		jsize len = (*env)->GetArrayLength(env, value);
		double *data = mxGetPr(result);
		double *jdata = (*env)->GetDoubleArrayElements(env, value, NULL);

		free(dims);

		// Copy data from Java array
		memcpy(data, jdata, len * sizeof(double));

		(*env)->ReleaseDoubleArrayElements(env, value, jdata, JNI_ABORT);

		return result;
	}
}

mxArray *getMatlabCharacterArray(JNIEnv *env, jobject javaArray) {
	jclass characterArrayClass = (*env)->FindClass(env, "org/simantics/matlablink/CharacterArray");
	jmethodID getCharactersMethod = (*env)->GetMethodID(env, characterArrayClass, "getCharacters", "()[C");
	jmethodID dimsMethod = (*env)->GetMethodID(env, characterArrayClass, "dims", "()[I");

	jcharArray value = (jcharArray)(*env)->CallObjectMethod(env, javaArray, getCharactersMethod);
	jintArray jdims = (*env)->CallObjectMethod(env, javaArray, dimsMethod);

	// Get vector dimentions
	jsize ndims = (*env)->GetArrayLength(env, jdims);
	jsize *jdimArray = (*env)->GetIntArrayElements(env, jdims, NULL);
	size_t *dims = (size_t*)malloc(ndims * sizeof(size_t));

	jint i;

	for (i = 0; i < ndims; i++) {
		dims[i] = jdimArray[i];
	}

	{
		mxArray *charArray = mxCreateCharArray(ndims, dims);

		jsize len = (*env)->GetArrayLength(env, value);
		jchar *bytes = (*env)->GetCharArrayElements(env, value, NULL);
		mxChar *bytesOut = mxGetChars(charArray);

		free(dims);
		(*env)->ReleaseIntArrayElements(env, jdims, jdimArray, JNI_ABORT);

		memcpy(bytesOut, bytes, len * sizeof(jchar));

		(*env)->ReleaseCharArrayElements(env, value, bytes, JNI_ABORT);

		return charArray;
	}
}

mxArray *getMatlabStructArray(JNIEnv *env, jobject value) {
	jclass structArrayClass = (*env)->FindClass(env, "org/simantics/matlablink/StructArray");
	jmethodID getFieldMethod = (*env)->GetMethodID(env, structArrayClass, "getField", "(ILjava/lang/String;)Lorg/simantics/matlablink/MatlabArray;");
	jmethodID getFieldsMethod = (*env)->GetMethodID(env, structArrayClass, "getFields", "()[Ljava/lang/String;");
	jmethodID dimsMethod = (*env)->GetMethodID(env, structArrayClass, "dims", "()[I");

	jobjectArray fields = (*env)->CallObjectMethod(env, value, getFieldsMethod);

	jint i, nfields = (*env)->GetArrayLength(env, fields);
	char **fieldNames = (char**)malloc(nfields * sizeof(char*));

	for (i = 0; i < nfields; i++) {
		jstring field = (jstring)(*env)->GetObjectArrayElement(env, fields, i);
		const char *chars = (*env)->GetStringUTFChars(env, field, NULL);
		fieldNames[i] = strdup(chars);
		(*env)->ReleaseStringUTFChars(env, field, chars);
	}

	{
		// Get vector dimentions
		jintArray jdims = (*env)->CallObjectMethod(env, value, dimsMethod);
		jsize ndims = (*env)->GetArrayLength(env, jdims);
		jsize *jdimArray = (*env)->GetIntArrayElements(env, jdims, NULL);
		size_t *dims = (size_t*)malloc(ndims * sizeof(size_t));
		mxArray *result;
		jsize len = ndims > 0 ? 1 : 0;

		for (i = 0; i < ndims; i++) {
			dims[i] = jdimArray[i];
			len = len * jdimArray[i];
		}

		result = mxCreateStructArray(ndims, dims, nfields, fieldNames);

		free(dims);

		for (i = 0; i < nfields; i++) {
			jstring field = (jstring)(*env)->GetObjectArrayElement(env, fields, i);
			int j;

			for (j = 0; j < len; j++) {
				jobject item = (*env)->CallObjectMethod(env, value, getFieldMethod, (jint)j, field);
				mxSetFieldByNumber(result, j, i, getMatlabArray(env, item));
			}
		}

		for (i = 0; i < nfields; i++)
			if (fieldNames[i] != NULL) free(fieldNames[i]);

		free(fieldNames);

		return result;
	}
}

mxArray *getMatlabCellArray(JNIEnv *env, jobject value) {
	jclass cellArrayClass = (*env)->FindClass(env, "org/simantics/matlablink/CellArray");
	jmethodID getCellValueMethod = (*env)->GetMethodID(env, cellArrayClass, "getCellValue", "()[Lorg/simantics/matlablink/MatlabArray;");
	jmethodID dimsMethod = (*env)->GetMethodID(env, cellArrayClass, "dims", "()[I");

	jobjectArray values = (*env)->CallObjectMethod(env, value, getCellValueMethod);
	jintArray jdims = (*env)->CallObjectMethod(env, value, dimsMethod);

	// Get vector dimentions
	jsize len = (*env)->GetArrayLength(env, values);
	jsize ndims = (*env)->GetArrayLength(env, jdims);
	jsize *jdimArray = (*env)->GetIntArrayElements(env, jdims, NULL);
	size_t *dims = (size_t*)malloc(ndims * sizeof(size_t));

	jint i;

	for (i = 0; i < ndims; i++) {
		dims[i] = jdimArray[i];
	}

	{
		// Create temporary Matlab array
		mxArray *matlabValue = mxCreateCellArray(ndims, dims);

		// Copy data from Java array
		for (i = 0; i < len; i++) {
			jobject item = (*env)->GetObjectArrayElement(env, values, i);
			mxArray *matlabItem = getMatlabArray(env, item);
			if (matlabItem != NULL) {
				mxSetCell(matlabValue, i, matlabItem);
			}
		}

		return matlabValue;
	}
}

mxArray *getMatlabArray(JNIEnv *env, jobject item) {
	jclass doubleArrayClass = (*env)->FindClass(env, "org/simantics/matlablink/DoubleArray");
	jclass characterArrayClass = (*env)->FindClass(env, "org/simantics/matlablink/CharacterArray");
	jclass cellArrayClass = (*env)->FindClass(env, "org/simantics/matlablink/CellArray");
	jclass structArrayClass = (*env)->FindClass(env, "org/simantics/matlablink/StructArray");

	mxArray *matlabItem = NULL;
	if ((*env)->IsInstanceOf(env, item, doubleArrayClass)) {
		matlabItem = getMatlabDoubleArray(env, item);
	}
	else if ((*env)->IsInstanceOf(env, item, characterArrayClass)) {
		matlabItem = getMatlabCharacterArray(env, item);
	}
	else if ((*env)->IsInstanceOf(env, item, cellArrayClass)) {
		matlabItem = getMatlabCellArray(env, item);
	}
	else if ((*env)->IsInstanceOf(env, item, structArrayClass)) {
		matlabItem = getMatlabStructArray(env, item);
	}

	return matlabItem;
}

// Generic arrays
JNIEXPORT jobject JNICALL Java_org_simantics_matlablink_Engine_getMatlabArrayImpl(JNIEnv *env, jobject thisObj, jlong engineID, jstring varName) {
	Engine *engine = (Engine *)engineID;

	if (varName == NULL) {
		throwIllegalArgumentException(env, "Null variable name given");
	}

	{
		mxArray *array = getMatlabVariable(env, engine, varName);

		if (array == NULL) {
			throwException(env, "java/lang/RuntimeException", "Variable not found");
			return NULL;
		}

		{
			jobject result = getJavaArray(env, array);

			mxDestroyArray(array);

			if (result == NULL) {
				throwException(env, "java/lang/RuntimeException", "Matlab variable type not supported");
				return NULL;
			}

			return result;
		}
	}
}

JNIEXPORT jint JNICALL Java_org_simantics_matlablink_Engine_setMatlabArrayImpl(JNIEnv *env, jobject thisObj, jlong engineID, jstring variableName, jobject value) {
	if (variableName == NULL) {
		throwIllegalArgumentException(env, "Null variable name given");
		return 0;
	}

	if (value == NULL) {
		throwIllegalArgumentException(env, "Null variable value given");
		return 0;
	}

	{
		Engine *engine = (Engine *)engineID;

		// Get value as mxArray
		mxArray *matlabValue = getMatlabArray(env, value);

		if (matlabValue == NULL) {
			throwException(env, "java/lang/RuntimeException", "Input not a suitable value for a Matlab array");
			return 0;
		}

		{
			// Set variable value
			int status = putMatlabVariable(env, engine, variableName, matlabValue);

			// Destroy array object
			mxDestroyArray(matlabValue);

			return status;
		}
	}
}

// Double arrays
JNIEXPORT jobject JNICALL Java_org_simantics_matlablink_Engine_getMatlabDoubleArrayImpl(JNIEnv *env, jobject thisObj, jlong engineID, jstring varName) {
	Engine *engine = (Engine *)engineID;

	if (varName == NULL) {
		throwIllegalArgumentException(env, "Null variable name given");
	}

	{
		mxArray *array = getMatlabVariable(env, engine, varName);

		if (array == NULL) {
			throwException(env, "java/lang/RuntimeException", "Variable not found");
			return NULL;
		}

		if (!mxIsDouble(array)) {
			throwException(env, "java/lang/RuntimeException", "Variable not a double array");
			return NULL;
		}

		{
			jdoubleArray result = getJavaDoubleArray(env, array);
			mxDestroyArray(array);
			return result;
		}
	}
}

JNIEXPORT jint JNICALL Java_org_simantics_matlablink_Engine_setMatlabDoubleArrayImpl(JNIEnv *env, jobject thisObj, jlong engineID, jstring variableName, jobject value) {
	if (variableName == NULL) {
		throwIllegalArgumentException(env, "Null variable name given");
		return 0;
	}

	if (value == NULL) {
		throwIllegalArgumentException(env, "Null variable value given");
		return 0;
	}

	{
		Engine *engine = (Engine *)engineID;

		// Get value as mxArray
		mxArray *matlabValue = getMatlabDoubleArray(env, value);

		// Set variable value
		int status = putMatlabVariable(env, engine, variableName, matlabValue);

		// Destroy array object
		mxDestroyArray(matlabValue);

		return status;
	}
}

// Cell arrays
JNIEXPORT jobject JNICALL Java_org_simantics_matlablink_Engine_getMatlabCellArrayImpl(JNIEnv *env, jobject thisObj, jlong engineID, jstring varName) {
	Engine *engine = (Engine *)engineID;

	if (varName == NULL) {
		throwIllegalArgumentException(env, "Null variable name given");
		return NULL;
	}

	{
		// Get variable value from Matlab
		mxArray *array = getMatlabVariable(env, engine, varName);

		if (array == NULL) {
			throwException(env, "java/lang/RuntimeException", "Variable not found");
			return NULL;
		}

		if (!mxIsCell(array)) {
			throwException(env, "java/lang/RuntimeException", "Variable not a cell array");
			return NULL;
		}

		{
			jobject result = getJavaCellArray(env, array);
			mxDestroyArray(array);

			if (result == NULL) {
				throwException(env, "java/lang/RuntimeException", "Matlab variable type not supported");
				return NULL;
			}

			return result;
		}
	}
}

JNIEXPORT jint JNICALL Java_org_simantics_matlablink_Engine_setMatlabCellArrayImpl(JNIEnv *env, jobject thisObj, jlong engineID, jstring variableName, jobject value) {
	if (variableName == NULL) {
		throwIllegalArgumentException(env, "Null variable name given");
		return 0;
	}

	if (value == NULL) {
		throwIllegalArgumentException(env, "Null variable value given");
		return 0;
	}

	{
		Engine *engine = (Engine *)engineID;

		// Get value as mxArray
		mxArray *matlabValue = getMatlabCellArray(env, value);

		// Set variable value
		int status = putMatlabVariable(env, engine, variableName, matlabValue);

		// Destroy array object
		mxDestroyArray(matlabValue);

		return status;
	}
}

// Character arrays
JNIEXPORT jobject JNICALL Java_org_simantics_matlablink_Engine_getMatlabCharacterArrayImpl(JNIEnv *env, jobject thisObj, jlong engineID, jstring varName) {
	Engine *engine = (Engine *)engineID;

	if (varName == NULL) {
		throwIllegalArgumentException(env, "Null variable name given");
		return NULL;
	}

	{
		// Get variable value from Matlab
		mxArray *array = getMatlabVariable(env, engine, varName);

		if (array == NULL) {
			throwException(env, "java/lang/RuntimeException", "Variable not found");
			return NULL;
		}

		if (!mxIsChar(array)) {
			throwException(env, "java/lang/RuntimeException", "Variable not a character array");
			return NULL;
		}

		{
			jobject result = getJavaCharacterArray(env, array);
			mxDestroyArray(array);

			if (result == NULL) {
				throwException(env, "java/lang/RuntimeException", "Matlab variable type not supported");
				return NULL;
			}

			return result;
		}
	}
}

JNIEXPORT jint JNICALL Java_org_simantics_matlablink_Engine_setMatlabCharacterArrayImpl(JNIEnv *env, jobject thisObj, jlong engineID, jstring variableName, jobject value) {
	if (variableName == NULL) {
		throwIllegalArgumentException(env, "Null variable name given");
		return 0;
	}

	if (value == NULL) {
		throwIllegalArgumentException(env, "Null variable value given");
		return 0;
	}

	{
		Engine *engine = (Engine *)engineID;

		// Get value as mxArray
		mxArray *matlabValue = getMatlabCharacterArray(env, value);

		// Set variable value
		int status = putMatlabVariable(env, engine, variableName, matlabValue);

		// Destroy array object
		mxDestroyArray(matlabValue);

		return status;
	}
}

// Struct arrays
JNIEXPORT jobject JNICALL Java_org_simantics_matlablink_Engine_getMatlabStructArrayImpl(JNIEnv *env, jobject thisObj, jlong engineID, jstring varName) {
	Engine *engine = (Engine *)engineID;

	if (varName == NULL) {
		throwIllegalArgumentException(env, "Null variable name given");
		return NULL;
	}

	{
		// Get variable value from Matlab
		mxArray *array = getMatlabVariable(env, engine, varName);

		if (array == NULL) {
			throwException(env, "java/lang/RuntimeException", "Variable not found");
			return NULL;
		}

		if (!mxIsStruct(array)) {
			throwException(env, "java/lang/RuntimeException", "Variable not a struct array");
			return NULL;
		}

		{
			jobject result = getJavaStructArray(env, array);
			mxDestroyArray(array);

			if (result == NULL) {
				throwException(env, "java/lang/RuntimeException", "Matlab variable type not supported");
				return NULL;
			}

			return result;
		}
	}
}

JNIEXPORT jint JNICALL Java_org_simantics_matlablink_Engine_setMatlabStructArrayImpl(JNIEnv *env, jobject thisObj, jlong engineID, jstring variableName, jobject value) {
	if (variableName == NULL) {
		throwIllegalArgumentException(env, "Null variable name given");
		return 0;
	}

	if (value == NULL) {
		throwIllegalArgumentException(env, "Null variable value given");
		return 0;
	}

	{
		Engine *engine = (Engine *)engineID;

		// Get value as mxArray
		mxArray *matlabValue = getMatlabStructArray(env, value);

		// Set variable value
		int status = putMatlabVariable(env, engine, variableName, matlabValue);

		// Destroy array object
		mxDestroyArray(matlabValue);

		return status;
	}
}

// Expression evaluation
JNIEXPORT jint JNICALL Java_org_simantics_matlablink_Engine_evaluateMatlabExpressionImpl(JNIEnv *env, jobject thisObj, jlong engineID, jstring expression) {
	Engine *engine = (Engine *)engineID;
	const char *expr = (*env)->GetStringUTFChars(env, expression, NULL);

	int status = engEvalString(engine, expr);

	(*env)->ReleaseStringUTFChars(env, expression, expr);

	return status;
}

//extern "C" DLL_EXPORT BOOL APIENTRY DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
BOOL __stdcall DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
    switch (fdwReason)
    {
        case DLL_PROCESS_ATTACH:
            // attach to process
            break;

        case DLL_PROCESS_DETACH:
            // detach from process
            break;

        case DLL_THREAD_ATTACH:
            // attach to thread
            break;

        case DLL_THREAD_DETACH:
            // detach from thread
            break;
    }
    return TRUE; // succesful
}
